/* zlib License
 *
 * Copyright (c) 2017-2026 Ed van Bruggen
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */
#include <ctype.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define EEVO_SHORTHAND
#include "eevo0.2.h"

struct Eevo_ eevo_nil  = { .t = EEVO_NIL };
struct Eevo_ eevo_true = { .t = EEVO_SYM, .v = { .s = "True" } };
struct Eevo_ eevo_void = { .t = EEVO_VOID };

#define BETWEEN(X, A, B)  ((A) <= (X) && (X) <= (B))
#define LEN(X)            (sizeof(X) / sizeof((X)[0]))

/* functions */
static void rec_add(EevoSt st, EevoRec rec, char *key, const Eevo val);
static Eevo eval_proc(EevoSt st, EevoRec env, Eevo f, Eevo args);


/*** Utilities ***/

/* return type of eevo value */
static Eevo
eevo_typeof(const EevoSt st, const EevoRec env, const Eevo args)
{
	eevo_arg_num(args, "Type", 1);
	int id = 0;
	for (int i = fst(args)->t; i > 1; i = i >> 1)
		id++;
	return st->types[id];
}

/* return named string for each type */
/* TODO loop through each type bit to print */
char *
eevo_type_str(EevoType t)
{
	switch (t) {
	case EEVO_VOID:  return "Void";
	case EEVO_NIL:   return "Nil";
	case EEVO_INT:   return "Int";
	case EEVO_DEC:   return "Dec";
	case EEVO_RATIO: return "Ratio";
	case EEVO_STR:   return "Str";
	case EEVO_SYM:   return "Sym";
	case EEVO_PRIM:  return "Prim";
	case EEVO_FORM:  return "Form";
	case EEVO_FUNC:  return "Func";
	case EEVO_MACRO: return "Macro";
	case EEVO_PAIR:  return "Pair";
	case EEVO_REC:   return "Rec";
	case EEVO_TYPE:  return "Type";
	case EEVO_RATIONAL: return "Rational";
	case EEVO_NUM:      return "Num";
	case EEVO_EXPR:     return "Expr";
	case EEVO_TEXT:     return "Text";
	case EEVO_PROC:     return "Proc";
	case EEVO_LIT:      return "Lit";
	case EEVO_LIST:     return "List";
	case EEVO_CALLABLE: return "Callable";
	case EEVO_FUNCTOR:  return "Functor";
	default:
		if (t &  EEVO_NUM) return "Num";
		return "Invalid";
	}
}

/* check if character can be a part of a symbol */
static int
is_sym(char c)
{
	/* TODO expand for all UTF: if c > 255 */
	return BETWEEN(c, 'a', 'z') || BETWEEN(c, 'A', 'Z') ||
	       BETWEEN(c, '0', '9') || strchr(EEVO_SYM_CHARS, c);
}

/* check if character can be a part of an operator */
static int
is_op(char c)
{
	return strchr(EEVO_OP_CHARS, c) != NULL;
}

/* check if character is start of a number */
/* TODO only if -/+ is not followed by a delim */
/* TODO use XOR '0' < 10 */
static int
isnum(const char *str)
{
	return isdigit(*str) || (*str == '.' &&  isdigit(str[1])) ||
	       ((*str == '-' || *str == '+') && (isdigit(str[1]) || str[1] == '.'));
}

/* skips:
 *  - spaces/tabs
 *  - newlines, if `skipnl` is true
 *  - comments, starting with ';' until end of line */
static void
skip_ws(EevoSt st, int skipnl)
{
	const char *s = skipnl ? " \t\n\r" : " \t";
	while (eevo_fget(st) && (strchr(s, eevo_fget(st)) || eevo_fget(st) == ';')) {
		/* skip contiguous white space */
		st->filec += strspn(st->file+st->filec, s);
		/* skip comments until newline */
		for (; eevo_fget(st) == ';'; eevo_finc(st))
			st->filec += strcspn(st->file+st->filec, "\n") - !skipnl;
	}
}

/* get length of list, if improper list return negative length */
int
eevo_lstlen(Eevo v)
{
	int len = 0;
	for (; v->t == EEVO_PAIR; v = rst(v))
		len++;
	return nilp(v) ? len : -(len + 1);
}

/* check if two values are equal */
static int
vals_eq(const Eevo a, const Eevo b)
{
	if (a->t & EEVO_NUM && b->t & EEVO_NUM) { /* NUMBERs */
		if (num(a) != num(b) || den(a) != den(b))
			return 0;
		return 1;
	}
	if (a->t != b->t)
		return 0;
	if (a->t == EEVO_PAIR) /* PAIR */
		return vals_eq(fst(a), fst(b)) && vals_eq(rst(a), rst(b));
	/* TODO function var names should not matter in comparison */
	if (a->t & (EEVO_FUNC | EEVO_MACRO)) /* FUNCTION, MACRO */
		return vals_eq(a->v.f.args, b->v.f.args) &&
		       vals_eq(a->v.f.body, b->v.f.body);
	if (a != b) /* PRIMITIVE, STRING, SYMBOL, NIL, VOID */
		return 0;
	return 1;
}

/* reduce fraction by modifying supplied numerator and denominator */
static void
frac_reduce(int *num, int *den)
{
	int a = abs(*num), b = abs(*den), c = a % b;
	while (c > 0) {
		a = b;
		b = c;
		c = a % b;
	}
	*num = *num / b;
	*den = *den / b;
}

/*** Stack ***/

static EevoStack *
new_region(size_t cap)
{
	size_t size = sizeof(EevoStack) + sizeof(uintptr_t)*cap;
	EevoStack *r;
	if (!(r = malloc(size))) /* TODO: page align */
		perror("; malloc"), exit(1);
	r->next = NULL;
	r->size = 0;
	r->cap  = cap;
	return r;
}

static void *
arena_alloc(EevoArena *a, size_t size_bytes)
{
	size_t size = (size_bytes + sizeof(uintptr_t) - 1)/sizeof(uintptr_t);

	/* Extend if requesting more than max capacity */
	while (a->end->size + size > a->end->cap && a->end->next)
		a->end = a->end->next;
	if (a->end->size + size > a->end->cap) {
		/* assert(!a->end->next); */
		a->end->next = new_region(size > EEVO_STACK_CAP ? size : EEVO_STACK_CAP);
		a->end = a->end->next;
	}

	/* Allocate new region */
	void *ret = &a->end->data[a->end->size];
	a->end->size += size;
	return ret;
}

static void
arena_free(EevoArena *a)
{
	for (EevoStack *r = a->beg; r;) {
		EevoStack *r0 = r;
		r = r->next;
		free(r0);
	}
	/* a->beg = NULL; */
	/* a->end = NULL; */
}

/*** Records ***/

/* return hashed number based on key */
static uint32_t
hash(const char *key)
{
	uint32_t h = 0;
	char c;
	while (h < UINT32_MAX && (c = *key++))
		h = h * 33 + c;
	return h;
}

/* create new empty rec with given capacity */
/* TODO rename rec ? */
static EevoRec
rec_new(EevoSt st, size_t cap, const EevoRec next)
{
	EevoRec rec = arena_alloc(&st->mem, sizeof(struct EevoRec_));
	rec->size = 0;
	rec->cap = cap;
	rec->items = arena_alloc(&st->mem, cap*sizeof(struct EevoEntry_));
	memset(rec->items, 0, cap*sizeof(rec->items));
	rec->next = next;
	return rec;
}

/* get entry in one record for the key */
static EevoEntry
entry_get(EevoRec rec, const char *key)
{
	int i = hash(key) % rec->cap;
	char *s;
	/* look for key starting at hash until empty entry is found */
	while ((s = rec->items[i].key)) {
		if (!strcmp(s, key))
			break;
		if (++i == rec->cap) /* loop back around if end is reached */
			i = 0;
	}
	return &rec->items[i]; /* returns entry if found or empty one to be filled */
}

/* get value of given key in each record */
static Eevo
rec_get(EevoRec rec, const char *key)
{
	for (; rec; rec = rec->next) {
		EevoEntry e = entry_get(rec, key);
		if (e->key)
			return e->val;
	}
	return NULL;
}

/* enlarge the record to ensure algorithm's efficiency */
/* TODO replace by implementing record as segmented array */
static void
rec_grow(EevoSt st, EevoRec rec)
{
	int i, ocap = rec->cap;
	EevoEntry oitems = rec->items;
	rec->cap *= EEVO_REC_FACTOR;
	rec->items = arena_alloc(&st->mem, rec->cap*sizeof(struct EevoEntry_));
	for (i = 0; i < ocap; i++) /* repopulate new record with old values */
		if (oitems[i].key)
			rec_add(st, rec, oitems[i].key, oitems[i].val);
}

/* create new key and value pair to the record */
static void
rec_add(EevoSt st, EevoRec rec, char *key, const Eevo val)
{
	EevoEntry e = entry_get(rec, key);
	e->val = val;
	if (!e->key) {
		e->key = key;
		/* grow record if it is more than half full */
		if (++rec->size > rec->cap / EEVO_REC_FACTOR)
			rec_grow(st, rec);
	}
}

/* add each vals[i] to rec with key args[i] */
static EevoRec
rec_extend(EevoSt st, EevoRec next, Eevo args, Eevo vals)
{
	Eevo arg, val;
	int argnum = EEVO_REC_FACTOR * eevo_lstlen(args);
	/* HACK need extra +1 for when argnum = 0 */
	EevoRec ret = rec_new(st, argnum > 0 ? argnum : -argnum + 1, next);
	for (; !nilp(args); args = rst(args), vals = rst(vals)) {
		if (args->t == EEVO_PAIR) {
			arg = fst(args);
			val = fst(vals);
		} else {
			arg = args;
			val = vals;
		}
		if (arg->t != EEVO_SYM)
			eevo_warnf("expected symbol for argument of function definition, "
			           "recieved '%s'",
			          eevo_type_str(arg->t));
		rec_add(st, ret, arg->v.s, val);
		if (args->t != EEVO_PAIR)
			break;
	}
	return ret;
}

/*** Type Constructors ***/

Eevo
eevo_val(EevoSt st, EevoType t)
{
	Eevo ret = arena_alloc(&st->mem, sizeof(struct Eevo_));
	ret->t = t;
	return ret;
}

Eevo
eevo_type(EevoSt st, EevoType t, char *name, Eevo func)
{
	Eevo ret = eevo_val(st, EEVO_INT);
	ret->t = EEVO_TYPE;
	ret->v.t = (EevoTypeVal){ .t = t, .name = name, .func = func };
	/* ret->t = EEVO_TYPE & t; */
	/* ret->v.f = (Func){ .name = name, .func = func }; */
	return ret;
}

Eevo
eevo_int(EevoSt st, int i)
{
	Eevo ret = eevo_val(st, EEVO_INT);
	num(ret) = i;
	den(ret) = 1;
	return ret;
}

Eevo
eevo_dec(EevoSt st, double d)
{
	Eevo ret = eevo_val(st, EEVO_DEC);
	num(ret) = d;
	den(ret) = 1;
	return ret;
}

Eevo
eevo_rat(EevoSt st, int num, int den)
{
	Eevo ret;
	if (den == 0)
		eevo_warn("division by zero");
	frac_reduce(&num, &den);
	if (den < 0) { /* simplify so only numerator is negative */
		den = abs(den);
		num = -num;
	}
	if (den == 1) /* simplify into integer if denominator is 1 */
		return eevo_int(st, num);
	ret = eevo_val(st, EEVO_RATIO);
	num(ret) = num;
	den(ret) = den;
	return ret;
}

/* TODO combine eevo_str and eevo_sym, replace st with intern hash */
Eevo
eevo_str(EevoSt st, char *s)
{
	Eevo ret;
	if ((ret = rec_get(st->strs, s)))
		return ret;
	ret = eevo_val(st, EEVO_STR);
	ret->v.s = s;
	rec_add(st, st->strs, s, ret);
	return ret;
}

Eevo
eevo_sym(EevoSt st, char *s)
{
	Eevo ret;
	if ((ret = rec_get(st->syms, s)))
		return ret;
	ret = eevo_val(st, EEVO_SYM);
	ret->v.s = s;
	rec_add(st, st->syms, s, ret);
	return ret;
}

Eevo
eevo_prim(EevoSt st, EevoType t, EevoPrim pr, char *name)
{
	Eevo ret = eevo_val(st, t);
	ret->v.pr.name = name;
	ret->v.pr.pr = pr;
	return ret;
}

Eevo
eevo_func(EevoSt st, EevoType t, char *name, Eevo args, Eevo body, EevoRec env)
{
	Eevo ret = eevo_val(st, t);
	ret->v.f.name = name;
	ret->v.f.args = args;
	ret->v.f.body = body;
	ret->v.f.env  = env;
	return ret;
}

/* TODO swap eevo_rec and rec_new */
/* TODO lazy eval, remove eval and store env in rec */
Eevo
eevo_rec(EevoSt st, EevoRec prev, const Eevo records)
{
	int cap;
	Eevo v, ret = eevo_val(st, EEVO_REC);
	if (!records)
		return ret->v.r = prev, ret;
	cap = EEVO_REC_FACTOR * eevo_lstlen(records);
	ret->v.r = rec_new(st, cap > 0 ? cap : -cap + 1, NULL);
	EevoRec r = rec_new(st, 4, prev);
	rec_add(st, r, "this", ret);
	for (Eevo cur = records; cur->t == EEVO_PAIR; cur = rst(cur))
		if (fst(cur)->t == EEVO_PAIR && ffst(cur)->t & (EEVO_SYM|EEVO_STR)) {
			if (!(v = eevo_eval(st, r, fst(rfst(cur)))))
				return NULL;
			rec_add(st, ret->v.r, ffst(cur)->v.s, v);
		} else if (fst(cur)->t == EEVO_SYM) {
			if (!(v = eevo_eval(st, r, fst(cur))))
				return NULL;
			rec_add(st, ret->v.r, fst(cur)->v.s, v);
		} else eevo_warn("Rec: missing key symbol or string");
	return ret;
}

Eevo
eevo_pair(EevoSt st, Eevo a, Eevo b)
{
	Eevo ret = eevo_val(st, EEVO_PAIR);
	fst(ret) = a;
	rst(ret) = b;
	return ret;
}

Eevo
eevo_list(EevoSt st, int n, ...)
{
	Eevo lst;
	va_list argp;
	va_start(argp, n);
	lst = eevo_pair(st, va_arg(argp, Eevo), Nil);
	for (Eevo cur = lst; n > 1; n--, cur = rst(cur))
		rst(cur) = eevo_pair(st, va_arg(argp, Eevo), Nil);
	va_end(argp);
	return lst;
}

/*** Read ***/

/* read first character of number to determine sign */
static int
read_sign(EevoSt st)
{
	switch (eevo_fget(st)) {
	case '-': eevo_finc(st); return -1;
	case '+': eevo_finc(st); return 1;
	default: return 1;
	}
}

/* return read integer */
static int
read_int(EevoSt st)
{
	char c;
	int ret = 0;
	for (; (c = eevo_fget(st)) && (isdigit(c) || c == '_'); eevo_finc(st))
		if (c != '_')
			ret = ret * 10 + (c - '0');
	return ret;
}

/* return integer read in any base: binary, octal, hexadecimal, etc */
/* TODO error on numbers higher than base (0b2, 0o9, etc) */
static Eevo
read_base(EevoSt st, int base)
{
	char c;
	int ret = 0;
	eevo_fincn(st, 2); /* skip the base signifier prefix (0b, 0o, 0x) */
	for (; (c = eevo_fget(st)) && (isxdigit(c) || c == '_'); eevo_finc(st))
		if (isdigit(c))
			ret = ret * base + (c - '0');
		else if (c != '_')
			ret = ret * base + (tolower(c) - 'a' + 10);
	return eevo_int(st, ret);
}

/* return read scientific notation */
static Eevo
read_sci(EevoSt st, double val, int isint)
{
	if (tolower(eevo_fget(st)) != 'e')
		goto finish;

	eevo_finc(st);
	double sign = read_sign(st) == 1 ? 10.0 : 0.1;
	for (int expo = read_int(st); expo--; val *= sign) ;

finish:
	if (isint)
		return eevo_int(st, val);
	return eevo_dec(st, val);
}

/* return read number */
static Eevo
read_num(EevoSt st)
{
	if (eevo_fget(st) == '0')
		switch (tolower(eevo_fgetat(st, 1))) {
		case 'b': return read_base(st, 2);
		case 'o': return read_base(st, 8);
		case 'x': return read_base(st, 16);
		}
	int sign = read_sign(st);
	int num = read_int(st);
	size_t oldc;
	switch (eevo_fget(st)) {
	case '/':
		if (!isnum(st->file + ++st->filec))
			eevo_warn("incorrect ratio format, no denominator found");
		return eevo_rat(st, sign * num, read_sign(st) * read_int(st));
	case '.':
		eevo_finc(st);
		oldc = st->filec;
		double d = (double) read_int(st);
		int size = st->filec - oldc;
		while (size--)
			d /= 10.0;
		return read_sci(st, sign * (num+d), 0);
	default:
		return read_sci(st, sign * num, 1);
	}
}

/* return character for escape */
static char
esc_char(char c)
{
	switch (c) {
	case 'n': return '\n';
	case 'r': return '\r';
	case 't': return '\t';
	case '\n': return ' ';
	case '\\':
	case '"':
	default:  return c;
	}
}

/* replace all encoded escape characters in string with their actual character */
static char *
esc_str(const char *s, int len, int do_esc)
{
	char *pos, *ret;
	if (!(ret = malloc((len+1) * sizeof(char))))
		perror("; malloc"), exit(1);
	for (pos = ret; pos-ret < len; pos++, s++)
		*pos = (*s == '\\' && do_esc) ? esc_char(*(++s)) : *s;
	*pos = '\0';
	return ret;
}

/* return read string or symbol, depending on eevo_fn */
static Eevo
read_str(EevoSt st, Eevo (*eevo_fn)(EevoSt, char*))
{
	int len = 0;
	char *s = st->file + ++st->filec; /* skip starting open quote */
	char endchar = eevo_fn == &eevo_str ? '"' : '~';
	/* get length of new escaped string */
	for (; eevo_fget(st) != endchar; eevo_finc(st), len++)
		if (!eevo_fget(st))
			eevo_warnf("reached end before closing %c", endchar);
		else if (eevo_fget(st) == '\\' && eevo_fgetat(st, -1) != '\\')
			eevo_finc(st); /* skip over break condition since it is escaped */
	eevo_finc(st); /* skip last closing quote */
	return eevo_fn(st, esc_str(s, len, eevo_fn == &eevo_str)); /* only escape strings */
}

/* return read symbol */
static Eevo
read_sym(EevoSt st, int (*is_char)(char))
{
	int len = 0;
	char *s = st->file + st->filec;
	for (; eevo_fget(st) && is_char(eevo_fget(st)); eevo_finc(st))
		len++; /* get length of new symbol */
	return eevo_sym(st, esc_str(s, len, 0));
}

/* return read list, pair, or improper list */
Eevo
read_pair(EevoSt st, char endchar)
{
	Eevo v, ret = eevo_pair(st, NULL, Nil);
	int skipnl = endchar != '\n';
	skip_ws(st, 1);
	/* if (!eevo_fget(st)) */
	/* 	return Nil; */
		/* eevo_warnf("reached end before closing '%c'", endchar); */
	/* TODO replace w/ strchr to also check for NULL and allow }} */
	/* !strchr(endchars, eevo_fget(st)) */
	for (Eevo pos = ret; eevo_fget(st) && eevo_fget(st) != endchar; pos = rst(pos)) {
		if (!(v = eevo_read(st)))
			return NULL;
		/* pair rest, end with non-nil (improper list) */
		if (v->t == EEVO_SYM && !strncmp(v->v.s, "...", 4)) {
			skip_ws(st, skipnl);
			if (!(v = eevo_read(st)))
				return NULL;
			rst(pos) = v;
			break;
		}
		rst(pos) = eevo_pair(st, v, Nil);
		/* if (v->t == EEVO_SYM && is_op(v->v.s[0])) { */
		/* 	is_infix = 1; */
		/* 	skip_ws(st, 1); */
		/* } else */
		skip_ws(st, skipnl);
	}
	skip_ws(st, skipnl);
	if (skipnl && eevo_fget(st) != endchar)
		eevo_warnf("did not find closing '%c'", endchar);
		/* eevo_warnf("found more than one element before closing '%c'", endchar); */
	eevo_finc(st);
	/* if (is_infix) */
	/* 	return fst(ret) = eevo_sym(st, "infix"), ret; */
	return rst(ret);
}

/* reads given string returning its eevo value */
Eevo
eevo_read_sexpr(EevoSt st)
{
	/* TODO merge w/ infix */
	/* TODO mk const global */
	static char *prefix[] = {
		"'",   "quote",
		"`",   "quasiquote",
		",@",  "unquote-splice", /* always check before , */
		",",   "unquote",
		"@",   "Func",
		"f\"", "strfmt",
		/* "?",   "try?", */
		/* "$",   "system!", */
		/* "-",   "negative", */
		/* "!",   "not?", */
	};
	skip_ws(st, 1); /* TODO dont skip nl in read */
	/* TODO replace w/ fget? */
	/* if == ] } ) etc say expected value before */
	if (strlen(st->file+st->filec) == 0) /* empty list */
		return Void;
	if (isnum(st->file+st->filec)) /* number */
		return read_num(st);
	if (eevo_fget(st) == '"') /* string */
		return read_str(st, eevo_str);
	if (eevo_fget(st) == '~') /* explicit symbol */
		return read_str(st, eevo_sym);
	for (int i = 0; i < LEN(prefix); i += 2) { /* character prefix */
		if (!strncmp(st->file+st->filec, prefix[i], strlen(prefix[i]))) {
			Eevo v;
			eevo_fincn(st, strlen(prefix[i]) - (prefix[i][1] == '"'));
			if (!(v = eevo_read(st))) return NULL;
			return eevo_list(st, 2, eevo_sym(st, prefix[i+1]), v);
		}
	}
	if (is_op(eevo_fget(st))) /* operators */
		return read_sym(st, &is_op);
	if (is_sym(eevo_fget(st))) /* symbols */
		return read_sym(st, &is_sym);
	if (eevo_fget(st) == '(') /* list */
		return eevo_finc(st), read_pair(st, ')');
	if (eevo_fget(st) == '[') /* list */
		return eevo_finc(st), eevo_pair(st, eevo_sym(st, "list"), read_pair(st, ']'));
	if (eevo_fget(st) == '{') { /* record */
		Eevo v; eevo_finc(st);
		if (!(v = read_pair(st, '}'))) return NULL;
		return eevo_pair(st, eevo_sym(st, "Rec"), v);
	}
	eevo_warnf("could not parse given input '%c' (%d)",
	          st->file[st->filec], (int)st->file[st->filec]);
}

/* read single value, made up of s-expression and optional syntax sugar */
Eevo
eevo_read(EevoSt st)
{
	Eevo v;
	if (!(v = eevo_read_sexpr(st)))
		return NULL;
	/* HACK find more general way to do this */
	while (eevo_fget(st) == '(' || eevo_fget(st) == ':' || eevo_fget(st) == '>' ||
	       eevo_fget(st) == '{')
		v = eevo_read_sugar(st, v);
	return v;
}

/* read extra syntax sugar on top of s-expressions */
Eevo
eevo_read_sugar(EevoSt st, const Eevo v)
{
	Eevo lst, w;
	if (eevo_fget(st) == '(') { /* func(x y) => (func x y) */
		/* FIXME @it(3) */
		eevo_finc(st);
		if (!(lst = read_pair(st, ')'))) return NULL;
		return eevo_pair(st, v, lst);
	} else if (eevo_fget(st) == '{') { /* rec{ key: value } => (recmerge rec { key: value }) */
		eevo_finc(st);
		if (!(lst = read_pair(st, '}'))) return NULL;
		return eevo_list(st, 3, eevo_sym(st, "recmerge"), v,
		                        eevo_pair(st, eevo_sym(st, "Rec"), lst));
	} else if (eevo_fget(st) == ':') {
		eevo_finc(st);
		switch (eevo_fget(st)) {
		case '(': /* proc:(a b c) => (map proc [a b c]) */
			eevo_finc(st);
			if (!(w = read_pair(st, ')'))) return NULL;
			return eevo_list(st, 3, eevo_sym(st, "map"), v,
			                        eevo_pair(st, eevo_sym(st, "list"), w));
		case ':': /* var::prop => (var 'prop) */
			eevo_finc(st);
			if (!(w = read_sym(st, &is_sym))) return NULL;
			return eevo_list(st, 2, v, eevo_list(st, 2, eevo_sym(st, "quote"), w));
		default: /* key: val => (key val) */
			skip_ws(st, 1);
			if (!(w = eevo_read(st))) return NULL;
			return eevo_list(st, 2, v, w);
		}
	} else if (eevo_fget(st) == '>' && eevo_fgetat(st, 1) == '>') {
		eevo_finc(st), eevo_finc(st);
		if (!(w = eevo_read(st)))
			eevo_warn("invalid UFCS");
		if (w->t != EEVO_PAIR)
			w = eevo_pair(st, w, Nil);
		return eevo_pair(st, fst(w), eevo_pair(st, v, rst(w)));
	}
	/* return eevo_pair(st, v, eevo_read(st)); */
	return v;
}

/* line reading synax sugar:
 *  - imply parenthesis around new lines
 *  - indented lines are sub-expressions
 *  - lines with single expression return just that expression */
Eevo
eevo_read_line(EevoSt st, int level)
{
	Eevo pos, ret;
	if (!(ret = read_pair(st, '\n'))) /* read line */
		return NULL;
	if (ret->t != EEVO_PAIR) /* force to be pair */
		ret = eevo_pair(st, ret, Nil);
	for (pos = ret; rst(pos)->t == EEVO_PAIR; pos = rst(pos)) ; /* get last pair */
	for (; eevo_fget(st); pos = rst(pos)) { /* read indented lines as sub-expressions */
		Eevo v;
		int newlevel = strspn(st->file+st->filec, "\t ");
		if (newlevel <= level)
			break;
		st->filec += newlevel;
		/* skip_ws(st, 1); */
		if (!(v = eevo_read_line(st, newlevel)))
			return NULL;
		if (!nilp(v))
			rst(pos) = eevo_pair(st, v, rst(pos));
	}
	return nilp(rst(ret)) ? fst(ret) : ret; /* if only 1 element in list, return just it */
}

/*** Evaluate ***/

/* evaluate each element of list */
/* TODO arg for eevo_eval or expand_macro */
/* TODO result list in args so can be given pointer to the same list to modify in place? */
Eevo
eevo_eval_list(EevoSt st, const EevoRec env, Eevo v)
{
	Eevo ret = eevo_pair(st, NULL, Nil), ev;
	for (Eevo cur = ret; !nilp(v); v = rst(v), cur = rst(cur)) {
		if (v->t != EEVO_PAIR) { /* last element in improper list */
			if (!(ev = eevo_eval(st, env, v)))
				return NULL;
			rst(cur) = ev;
			return rst(ret);
		}
		if (!(ev = eevo_eval(st, env, fst(v))))
			return NULL;
		rst(cur) = eevo_pair(st, ev, Nil);
	}
	return rst(ret);
}

/* evaluate all elements of list returning last */
Eevo
eevo_eval_body(EevoSt st, EevoRec env, Eevo body)
{
	Eevo ret = Void;
	while (body->t == EEVO_PAIR) {
		if (nilp(rst(body)) && fst(body)->t == EEVO_PAIR) { /* func call is last, do tail call */
			Eevo f, args;
			if (!(f = eevo_eval(st, env, ffst(body))))
				return NULL;
			if (f->t != EEVO_FUNC)
				return eval_proc(st, env, f, rfst(body));
			eevo_arg_num(rfst(body), f->v.f.name ? f->v.f.name : "anon",
			            eevo_lstlen(f->v.f.args));
			if (!(args = eevo_eval_list(st, env, rfst(body))))
				return NULL;
			if (!(env = rec_extend(st, f->v.f.env, f->v.f.args, args)))
				return NULL;
			/* continue loop from body of func call */
			body = f->v.f.body;
			continue;
		} else if (!(ret = eevo_eval(st, env, fst(body))))
			return NULL;
		body = rst(body);
	}
	return ret;
}

static void
prepend_bt(EevoSt st, EevoRec env, Eevo f)
{
	if (!f->v.f.name) /* no need to record anonymous functions */
		return;
	for (; env->next; env = env->next) ; /* bt var located at base env */
	EevoEntry e = entry_get(env, "bt");
	if (e->val->t == EEVO_PAIR && fst(e->val)->t == EEVO_SYM &&
	    !strncmp(f->v.f.name, fst(e->val)->v.s, strlen(fst(e->val)->v.s)))
		return; /* don't record same function on recursion */
	e->val = eevo_pair(st, eevo_sym(st, f->v.f.name), e->val);
}

/* evaluate procedure f with arguments */
static Eevo
eval_proc(EevoSt st, EevoRec env, Eevo f, Eevo args)
{
	Eevo ret;
	EevoRec fenv;
	/* evaluate function and primitive arguments before being passed */
	switch (f->t) {
	case EEVO_PRIM:
		if (!(args = eevo_eval_list(st, env, args)))
			return NULL;
		/* FALLTHROUGH */
	case EEVO_FORM:
		return (*f->v.pr.pr)(st, env, args);
	case EEVO_FUNC:
		if (!(args = eevo_eval_list(st, env, args)))
			return NULL;
		/* FALLTHROUGH */
	case EEVO_MACRO:
		eevo_arg_num(args, f->v.f.name ? f->v.f.name : "anon", eevo_lstlen(f->v.f.args));
		if (!(fenv = rec_extend(st, f->v.f.env, f->v.f.args, args)))
			return NULL;
		if (!(ret = eevo_eval_body(st, fenv, f->v.f.body)))
			return prepend_bt(st, env, f), NULL;
		if (f->t == EEVO_MACRO) /* TODO remove w/ expand_macro */
			ret = eevo_eval(st, env, ret);
		return ret;
	case EEVO_REC:
		if (!(args = eevo_eval_list(st, env, args)))
			return NULL;
		eevo_arg_num(args, "record", 1);
		eevo_arg_type(fst(args), "record", EEVO_SYM);
		if (!(ret = rec_get(f->v.r, fst(args)->v.s)) &&
		    !(ret = rec_get(f->v.r, "else")))
			eevo_warnf("could not find element '%s' in record", fst(args)->v.s);
		return ret;
	case EEVO_TYPE:
		if (f->v.t.func)
			return eval_proc(st, env, f->v.t.func, args);
		eevo_warnf("could not convert to type '%s'", f->v.t.name);
	case EEVO_PAIR: // TODO eval each element as func w/ args: body
	default:
		eevo_warnf("attempt to evaluate non procedural type '%s' (%s)",
				eevo_type_str(f->t), eevo_print(f));
	}
}

/* evaluate given value */
Eevo
eevo_eval(EevoSt st, const EevoRec env, const Eevo v)
{
	Eevo f;
	switch (v->t) {
	case EEVO_SYM:
		if ((f = rec_get(env, v->v.s)))
			return f;
		if (v == eevo_sym(st, "ENV"))
			return eevo_rec(st, env, NULL);
		eevo_warnf("could not find symbol '%s'", v->v.s);
	case EEVO_PAIR:
		if (!(f = eevo_eval(st, env, fst(v))))
			return NULL;
		return eval_proc(st, env, f, rst(v));
	default:
		return v;
	}
}

/*** Print ***/

/* determine size of string to be printed */
size_t
print_size(const Eevo v)
{
	int len = 0;
	switch (v->t) {
	case EEVO_VOID:  return 5;
	case EEVO_NIL:   return 4;
	case EEVO_INT:   return snprintf(NULL, 0, "%d", (int)num(v)) + 1;
	case EEVO_DEC:   return snprintf(NULL, 0, "%.15G", num(v)) + 3;
	case EEVO_RATIO: return snprintf(NULL, 0, "%d/%d", (int)num(v), (int)den(v)) + 1;
	case EEVO_STR:
	case EEVO_SYM:   return strlen(v->v.s) + 1;
	case EEVO_FUNC:
	case EEVO_MACRO:
		if (!v->v.f.name) return 5; /* anon */
		return strlen(v->v.f.name) + 1;
	case EEVO_PRIM:
	case EEVO_FORM: return strlen(v->v.pr.name);
	case EEVO_TYPE: return strlen(v->v.t.name);
	case EEVO_REC:
		for (EevoRec r = v->v.r; r; r = r->next)
			for (int i = 0, c = 0; c < r->size; i++)
				if (r->items[i].key) {
					len += strlen(r->items[i].key) +
					       print_size(r->items[i].val) + 2;
					c++;
				}
		return len;
	case EEVO_PAIR: return print_size(fst(v)) + print_size(rst(v)) + 1;
	default:        return 0;
	}
}

/* Convert record to string stored in ret */
static void
print_rec(char *ret, const EevoRec rec)
{
	int len = 0;
	for (EevoRec r = rec; r; r = r->next)
		for (int i = 0, c = 0; c < r->size; i++)
			if (r->items[i].key) {
				int olen = len;
				char *key = r->items[i].key;
				char *val = eevo_print(r->items[i].val);
				len += strlen(key) + strlen(val) + 2;
				snprintf(ret + strlen(ret), len-olen, "%s:%s", key, val);
				free(val);
				c++;
			}
}

/* Convert eevo value to string to be printed
 *   returned string needs to be freed after use */
char *
eevo_print(const Eevo v)
{
	int size = print_size(v);
	char *head, *tail, *ret = calloc(size, sizeof(char));
	switch (v->t) {
	case EEVO_VOID:
		strcat(ret, "Void");
		break;
	case EEVO_NIL:
		strcat(ret, "Nil");
		break;
	case EEVO_INT:
		snprintf(ret, size, "%d", (int)num(v));
		break;
	case EEVO_DEC:
		snprintf(ret, size, "%.15G", num(v));
		if (num(v) == (int)num(v))
			strcat(ret, ".0");
		break;
	case EEVO_RATIO:
		snprintf(ret, size, "%d/%d", (int)num(v), (int)den(v));
		break;
	case EEVO_STR:
	case EEVO_SYM:
		strcat(ret, v->v.s);
		break;
	case EEVO_FUNC:
	case EEVO_MACRO:
		if (!v->v.f.name) {
			strcat(ret, "anon");
			break;
		}
		strcat(ret, v->v.f.name);
		break;
	case EEVO_PRIM:
	case EEVO_FORM:
		strcat(ret, v->v.pr.name);
		break;
	case EEVO_TYPE:
		strcat(ret, v->v.t.name);
		break;
	case EEVO_REC:
		print_rec(ret, v->v.r);
		break;
	case EEVO_PAIR:
		head = eevo_print(fst(v));
		tail = nilp(rst(v)) ? "" : eevo_print(rst(v));
		snprintf(ret, size, "%s%s", head, tail);
		free(head);
		if (!nilp(rst(v)))
			free(tail);
		break;
	default:
		free(ret);
		eevo_warnf("could not print type '%s'", eevo_type_str(v->t));
	}
	return ret;
}


/*** Environment ***/

/* add new variable of name key and value v to the given environment */
void
eevo_env_add(EevoSt st, char *key, const Eevo v)
{
	rec_add(st, st->env, key, v);
}

/* initialise eevo's state and global environment */
EevoSt
eevo_env_init(size_t cap)
{
	EevoArena mem;
	mem.end = new_region(cap);
	mem.beg = mem.end;

	EevoSt st = arena_alloc(&mem, sizeof(struct EevoSt_));

	st->mem = mem;

	st->file = NULL;
	st->filec = 0;

	/* TODO intern (memorize) all types, including stateless func calls */
	st->strs = rec_new(st, cap, NULL);
	st->syms = rec_new(st, cap, NULL);

	st->env = rec_new(st, cap, NULL);
	eevo_env_add(st, "True", True);
	eevo_env_add(st, "Nil", Nil);
	eevo_env_add(st, "Void", Void);
	eevo_env_add(st, "bt", Nil);
	eevo_env_add(st, "version", eevo_str(st, "0.2.1"));

	/* Types */
	st->types[0]  = eevo_type(st, EEVO_VOID,  "TVoid", NULL);
	st->types[1]  = eevo_type(st, EEVO_NIL,   "TNil",  NULL);
	st->types[2]  = eevo_type(st, EEVO_INT,   "Int",   NULL);
	st->types[3]  = eevo_type(st, EEVO_DEC,   "Dec",   NULL);
	st->types[4]  = eevo_type(st, EEVO_RATIO, "Ratio", NULL);
	st->types[5]  = eevo_type(st, EEVO_STR,   "Str",   NULL);
	st->types[6]  = eevo_type(st, EEVO_SYM,   "Sym",   NULL);
	st->types[7]  = eevo_type(st, EEVO_PRIM,  "Prim",  NULL);
	st->types[8]  = eevo_type(st, EEVO_FORM,  "Form",  NULL);
	st->types[9]  = eevo_type(st, EEVO_FUNC,  "Func",  NULL);
	st->types[10] = eevo_type(st, EEVO_MACRO, "Macro", NULL);
	st->types[11] = eevo_type(st, EEVO_PAIR,  "Pair",  NULL);
	/* Eevo lst = eevo_sym(st, "lst"); */
	/* Eevo List = eevo_func(EEVO_FUNC, "List", lst, eevo_list(st, 1, lst), st->env); */
	/* st->types[11] = eevo_type(st, EEVO_PAIR | EEVO_VOID,  "List",  List); */
	st->types[12] = eevo_type(st, EEVO_REC, "Rec",
	                          eevo_prim(st, EEVO_FORM, eevo_rec,    "Rec"));
	st->types[13] = eevo_type(st, EEVO_TYPE, "Type",
	                          eevo_prim(st, EEVO_PRIM, eevo_typeof, "Type"));
	for (int i = 0; i < LEN(st->types); i++)
		eevo_env_add(st, st->types[i]->v.t.name, st->types[i]);
		/* TODO define type predicate functions here (nil?, string?, etc) */

	/* eevo_env_lib(st, libs); */

	return st;
}

void
eevo_free(EevoSt st)
{
	arena_free(&st->mem);
}

char eevo_core[];

/* load lib from string into environment */
Eevo
eevo_env_lib(EevoSt st)
{
	Eevo parsed, expr, ret;
	char *file = st->file;
	size_t filec = st->filec;
	st->file = eevo_core;
	st->filec = 0;
	skip_ws(st, 1);
	parsed = eevo_pair(st, eevo_sym(st, "do"), Nil);
	for (Eevo pos = parsed; eevo_fget(st) && (expr = eevo_read_line(st, 0)); pos = rst(pos))
		rst(pos) = eevo_pair(st, expr, Nil);
	ret = eevo_eval_body(st, st->env, rst(parsed));
	st->file = file;
	st->filec = filec;
	return ret;
}
/* zlib License
 *
 * Copyright (c) 2017-2026 Ed van Bruggen
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */
#include <dlfcn.h>
#include <fcntl.h>
#include <limits.h>
#include <unistd.h>

/* count number of parenthesis, brackets, and curly braces */
/* FIXME makes reading O(2N) replace w/ better counting sys */
static int
count_parens(char *s, int len)
{
	int pcount = 0, bcount = 0, ccount = 0;
	for (int i = 0; i < len && s[i]; i++) {
		switch (s[i]) {
			case '(': pcount++; break;
			case '[': bcount++; break;
			case '{': ccount++; break;
			case ')': pcount--; break;
			case ']': bcount--; break;
			case '}': ccount--; break;
		}
	}
	if (pcount)
		return pcount;
	if (bcount)
		return bcount;
	return ccount;
}

/* return string containing contents of file name */
static char *
read_file(char *fname)
{
	char buf[BUFSIZ], *file = NULL;
	int len = 0, n, fd, parens = 0;
	if (!fname) /* read from stdin if no file given */
		fd = 0;
	else if ((fd = open(fname, O_RDONLY)) < 0)
		eevo_warnf("could not find file '%s'", fname);
	while ((n = read(fd, buf, sizeof(buf))) > 0) {
		file = realloc(file, len + n + 1);
		if (!file) perror("; realloc"), exit(1);
		memcpy(file + len, buf, n);
		len += n;
		file[len] = '\0';
		if (fd == 0 && (parens += count_parens(buf, n)) <= 0)
			break;
	}
	if (fd) /* close file if not stdin */
		close(fd);
	if (n < 0)
		eevo_warnf("could not read file '%s'", fname);
	return file;
}

/* write all arguemnts to given file, or stdout/stderr, without newline */
/* first argument is file name, second is option to append file */
static Eevo
prim_write(EevoSt st, EevoRec env, Eevo args)
{
	FILE *f;
	const char *mode = "w";
	eevo_arg_min(args, "write", 2);

	/* if second argument is true, append file don't write over */
	if (!nilp(snd(args)))
		mode = "a";
	/* first argument can either be the symbol stdout or stderr, or the file as a string */
	if (fst(args)->t == EEVO_SYM)
		f = !strncmp(fst(args)->v.s, "stdout", 7) ? stdout : stderr;
	else if (fst(args)->t != EEVO_STR)
		eevo_warnf("write: expected file name as string, received %s",
		           eevo_type_str(fst(args)->t));
	else if (!(f = fopen(fst(args)->v.s, mode)))
		eevo_warnf("write: could not load file '%s'", fst(args)->v.s);
	if (f == stderr && strncmp(fst(args)->v.s, "stderr", 7)) /* validate stderr symbol */
		eevo_warn("write: expected file name as string, or symbol stdout/stderr");

	for (args = rrst(args); !nilp(args); args = rst(args)) {
		char *out = eevo_print(fst(args));
		fputs(out, f);
		free(out);
	}

	if (f == stdout || f == stderr) /* clean up */
		fflush(f);
	else
		fclose(f);
	return Void;
}

/* return string of given file or read from stdin */
static Eevo
prim_read(EevoSt st, EevoRec env, Eevo args)
{
	char *file, *fname = NULL; /* read from stdin by default */
	eevo_arg_max(args, "read", 1);
	if (eevo_lstlen(args) == 1) { /* if file name given as string, read it */
		eevo_arg_type(fst(args), "read", EEVO_STR);
		fname = fst(args)->v.s;
	}
	if (!(file = read_file(fname)))
		return Nil;
	return eevo_str(st, file);
}

/* parse string as eevo expression, return 'quit if given no arguments */
static Eevo
prim_parse(EevoSt st, EevoRec env, Eevo args)
{
	Eevo ret, expr;
	char *file = st->file;
	size_t filec = st->filec;
	eevo_arg_num(args, "parse", 1);
	expr = fst(args);
	if (nilp(expr))
		return eevo_sym(st, "quit");
	eevo_arg_type(expr, "parse", EEVO_STR);
	st->file = expr->v.s;
	st->filec = 0;
	ret = eevo_pair(st, eevo_sym(st, "do"), Nil);
	for (Eevo pos = ret; eevo_fget(st) && (expr = eevo_read_line(st, 0)); pos = rst(pos))
		rst(pos) = eevo_pair(st, expr, Nil);
	st->file = file;
	st->filec = filec;
	if (rst(ret)->t == EEVO_PAIR && nilp(rrst(ret)))
		return snd(ret); /* if only 1 expression parsed, return just it */
	return ret;
}

/* loads dynamic C library */
/* TODO load static libs as well by passing NULL to dlopen */
static Eevo
form_load(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "load", 2);
	eevo_arg_type(fst(args), "load", EEVO_TEXT);
	eevo_arg_type(snd(args), "load", EEVO_TEXT);
	char *lib = fst(args)->v.s;
	char *name = snd(args)->v.s;

	/* Load dynamic library into libh */
	void *libh;
	if (!(libh = dlopen(lib, RTLD_LAZY)))
		eevo_warnf("load: could not load '%s':\n; %s", lib, dlerror());

	/* Get primitive from library */
	dlerror(); /* Clear error */
	EevoPrim pr = (EevoPrim)dlsym(libh, name);
	if (dlerror())
		eevo_warnf("load: could not load '%s' from '%s':\n; %s", name, lib, dlerror());
	return eevo_prim(st, EEVO_FORM, pr, name);
}

void
eevo_env_io(EevoSt st)
{
	eevo_env_prim(write);
	eevo_env_prim(read);
	eevo_env_prim(parse);
	eevo_env_form(load);
}
/* zlib License
 *
 * Copyright (c) 2017-2026 Ed van Bruggen
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

/* return first element of list */
static Eevo
prim_fst(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "fst", 1);
	eevo_arg_type(fst(args), "fst", EEVO_PAIR);
	return ffst(args);
}

/* return elements of a list after the first */
static Eevo
prim_rst(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "rst", 1);
	eevo_arg_type(fst(args), "rst", EEVO_PAIR);
	return rfst(args);
}

/* return new pair */
static Eevo
prim_Pair(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "Pair", 2);
	return eevo_pair(st, fst(args), snd(args));
}

/* do not evaluate argument */
static Eevo
form_quote(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "quote", 1);
	return fst(args);
}

/* evaluate argument given */
static Eevo
prim_eval(EevoSt st, EevoRec env, Eevo args)
{
	Eevo v;
	eevo_arg_num(args, "eval", 1);
	return (v = eevo_eval(st, st->env, fst(args))) ? v : Void;
}

/* test equality of all values given */
static Eevo
prim_eq(EevoSt st, EevoRec env, Eevo args)
{
	if (nilp(args))
		return True;
	for (; !nilp(rst(args)); args = rst(args))
		if (!vals_eq(fst(args), snd(args)))
			return Nil;
	/* return nilp(fst(args)) ? True : fst(args); */
	return True;
}

/* evaluates and returns first expression with a true conditional */
static Eevo
form_cond(EevoSt st, EevoRec env, Eevo args)
{
	Eevo v, cond;
	for (v = args; !nilp(v); v = rst(v))
		if (!(cond = eevo_eval(st, env, ffst(v))))
			return NULL;
		else if (!nilp(cond)) /* TODO incorporate else directly into cond */
			return eevo_eval_body(st, env, rfst(v));
	return Void;
}

/* return type of eevo value */
static Eevo
prim_typeof(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "typeof", 1);
	return eevo_str(st, eevo_type_str(fst(args)->t));
}

/* return record of properties for given procedure */
static Eevo
prim_procprops(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "procprops", 1);
	Eevo proc = fst(args);
	EevoRec ret = rec_new(st, 6, NULL); /* TODO mv eevo_rec here */
	switch (proc->t) {
	case EEVO_FORM:
	case EEVO_PRIM:
		rec_add(st, ret, "name", eevo_sym(st, proc->v.pr.name));
		break;
	case EEVO_FUNC:
	case EEVO_MACRO:
		rec_add(st, ret, "name", eevo_sym(st, proc->v.f.name ? proc->v.f.name : "anon"));
		rec_add(st, ret, "args", proc->v.f.args);
		rec_add(st, ret, "body", proc->v.f.body);
		/* rec_add(st, ret, "env", proc->v.f.env); */
		break;
	default:
		eevo_warnf("procprops: expected Proc, received '%s'", eevo_type_str(proc->t));
	}
	return eevo_rec(st, ret, NULL);
}

/* creates new eevo function */
static Eevo
form_Func(EevoSt st, EevoRec env, Eevo args)
{
	Eevo params, body;
	eevo_arg_min(args, "Func", 1);
	if (nilp(rst(args))) { /* if only 1 argument is given, auto fill func parameters */
		params = eevo_pair(st, eevo_sym(st, "it"), Nil);
		body = args;
	} else {
		params = fst(args);
		body = rst(args);
	}
	return eevo_func(st, EEVO_FUNC, NULL, params, body, env);
}

/* creates new eevo defined macro */
static Eevo
form_Macro(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_min(args, "Macro", 1);
	Eevo ret = form_Func(st, env, args);
	ret->t = EEVO_MACRO;
	return ret;
}

/* display message and return error */
static Eevo
prim_error(EevoSt st, EevoRec env, Eevo args)
{
	char *msg;
	/* TODO have error auto print function name that was pre-defined */
	eevo_arg_min(args, "error", 2);
	eevo_arg_type(fst(args), "error", EEVO_SYM);
	if (!(msg = eevo_print(rst(args))))
		return NULL;
	fprintf(stderr, "; eevo: error: %s: %s\n", fst(args)->v.s, msg);
	free(msg);
	return NULL;
}

/** Records **/

/* merge second record into first record, without mutation */
static Eevo
prim_recmerge(EevoSt st, EevoRec env, Eevo args)
{
	Eevo ret = eevo_val(st, EEVO_REC); /* TODO use eevo_rec */
	eevo_arg_num(args, "recmerge", 2);
	eevo_arg_type(fst(args), "recmerge", EEVO_REC);
	eevo_arg_type(snd(args), "recmerge", EEVO_REC);
	ret->v.r = rec_new(st, snd(args)->v.r->size*EEVO_REC_FACTOR, fst(args)->v.r);
	for (EevoRec r = snd(args)->v.r; r; r = r->next)
		for (int i = 0, c = 0; c < r->size; i++)
			if (r->items[i].key)
				c++, rec_add(st, ret->v.r,
				             r->items[i].key, r->items[i].val);
	return ret;
}

/* retrieve list of every entry in given record */
static Eevo
prim_records(EevoSt st, EevoRec env, Eevo args)
{
	Eevo ret = Nil;
	eevo_arg_num(args, "records", 1);
	eevo_arg_type(fst(args), "records", EEVO_REC);
	for (EevoRec r = fst(args)->v.r; r; r = r->next)
		for (int i = 0, c = 0; c < r->size; i++)
			if (r->items[i].key) {
				Eevo entry = eevo_pair(st, eevo_sym(st, r->items[i].key),
				                                 r->items[i].val);
				ret = eevo_pair(st, entry, ret);
				c++;
			}
	return ret;
}

/* creates new variable of given name and value
 * if pair is given as name of variable, creates function with the first value as the
 * function name and the rest as the function arguments */
/* TODO if var not func error if more than 2 args */
static Eevo
form_def(EevoSt st, EevoRec env, Eevo args)
{
	Eevo sym, val;
	eevo_arg_min(args, "def", 1);
	if (fst(args)->t == EEVO_PAIR) { /* create function if given argument list */
		sym = ffst(args); /* first element of argument list is function name */
		if (sym->t != EEVO_SYM)
			eevo_warnf("def: expected symbol for function name, received '%s'",
			          eevo_type_str(sym->t));
		val = eevo_func(st, EEVO_FUNC, sym->v.s, rfst(args), rst(args), env);
	} else if (fst(args)->t == EEVO_SYM) { /* create variable */
		sym = fst(args); /* if only symbol given, make it self evaluating */
		val = nilp(rst(args)) ? sym : eevo_eval(st, env, snd(args));
	} else eevo_warn("def: incorrect format, no variable name found");
	if (!val)
		return NULL;
	/* set procedure name if it was previously anonymous */
	if (val->t & (EEVO_FUNC|EEVO_MACRO) && !val->v.f.name)
		val->v.f.name = sym->v.s; /* TODO some bug here */
	rec_add(st, env, sym->v.s, val);
	return Void;
}

/* TODO fix crashing if try to undefine builtin */
static Eevo
form_undefine(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_min(args, "undefine!", 1);
	eevo_arg_type(fst(args), "undefine!", EEVO_SYM);
	for (EevoRec r = env; r; r = r->next) {
		EevoEntry e = entry_get(r, fst(args)->v.s);
		if (e->key) {
			e->key = NULL;
			/* TODO eevo_free(e->val); */
			return Void;
		}
	}
	eevo_warnf("undefine!: could not find symbol %s to undefine", fst(args)->v.s);
}

static Eevo
form_definedp(EevoSt st, EevoRec env, Eevo args)
{
	EevoEntry e = NULL;
	eevo_arg_min(args, "defined?", 1);
	eevo_arg_type(fst(args), "defined?", EEVO_SYM);
	for (EevoRec r = env; r; r = r->next) {
		e = entry_get(r, fst(args)->v.s);
		if (e->key)
			break;
	}
	return (e && e->key) ? True : Nil;
}

/* check if value is a member of a collection */
static Eevo
prim_in(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_min(args, "in", 2);
	Eevo key = fst(args);
	Eevo col = snd(args);
	switch (col->t) {
	case EEVO_TYPE:
		return key->t & col->v.t.t ? True : Nil;
	case EEVO_STR:
	case EEVO_SYM:
		eevo_arg_type(key, "in", EEVO_TEXT);
		return strstr(col->v.s, key->v.s) ? True : Nil;
	case EEVO_PAIR:
		/* TODO: work on sublists like Str ? */
		for (Eevo cur = col; cur->t == EEVO_PAIR; cur = rst(cur))
			if (vals_eq(key, fst(cur)))
				return True;
		return Nil;
	case EEVO_REC:
		eevo_arg_type(key, "in", EEVO_TEXT);
		for (EevoRec r = col->v.r; r; r = r->next) {
			EevoEntry e = entry_get(r, key->v.s);
			if (e->key)
				return True;
		}
		return Nil;
	case EEVO_PRIM:
	case EEVO_FORM:
	case EEVO_FUNC:
	case EEVO_MACRO:
		/* TODO: if key is valid input */
	default:
		eevo_warnf("in: expected collection, recieved %s", eevo_type_str(col->t));
	}
}

void
eevo_env_core(EevoSt st)
{
	eevo_env_prim(fst);
	eevo_env_prim(rst);
	st->types[11]->v.t.func = eevo_prim(st, EEVO_PRIM, prim_Pair, "Pair");
	eevo_env_form(quote);
	eevo_env_prim(eval);
	eevo_env_name_prim(=, eq);
	eevo_env_form(cond);
	eevo_env_add(st, "do", eevo_prim(st, EEVO_FORM, eevo_eval_body, "do"));

	eevo_env_prim(typeof);
	eevo_env_prim(procprops);
	st->types[9]->v.t.func  = eevo_prim(st, EEVO_FORM, form_Func,  "Func");
	st->types[10]->v.t.func = eevo_prim(st, EEVO_FORM, form_Macro, "Macro");
	eevo_env_prim(error);

	eevo_env_prim(recmerge);
	eevo_env_prim(records);
	eevo_env_form(def);
	eevo_env_name_form(undefine!, undefine);
	eevo_env_name_form(defined?, definedp);
	eevo_env_prim(in);
}
/* zlib License
 *
 * Copyright (c) 2017-2026 Ed van Bruggen
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

typedef Eevo (*MkFn)(EevoSt, char*);

/* TODO string tib: lower upper capitalize strpos strsub skipto snipto (python: dir(str))*/

/* convert all args to a string */
static Eevo
prim_Str(EevoSt st, EevoRec env, Eevo args)
{
	char *s;
	if (!(s = eevo_print(args)))
		return NULL;
	return eevo_str(st, s);
}

/* convert all args to a symbol */
static Eevo
prim_Sym(EevoSt st, EevoRec env, Eevo args)
{
	char *s;
	if (!(s = eevo_print(args)))
		return NULL;
	return eevo_sym(st, s);
}

static Eevo
prim_strlen(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_min(args, "strlen", 1);
	eevo_arg_type(fst(args), "strlen", EEVO_STR | EEVO_SYM);
	return eevo_int(st, strlen(fst(args)->v.s));
}

/* perform interpolation on explicit string, evaluating anything inside curly braces */
/* FIXME nested strings shouldn't need to be escaped*/
static Eevo
form_strfmt(EevoSt st, EevoRec env, Eevo args)
{
	char *ret, *str;
	int ret_len, ret_cap, pos = 0;
	Eevo v;
	eevo_arg_num(args, "strfmt", 1);
	eevo_arg_type(fst(args), "strfmt", EEVO_STR);

	str = fst(args)->v.s;
	ret_len = strlen(str), ret_cap = 2*ret_len;
	if (!(ret = malloc(sizeof(char) * ret_cap)))
		perror("; malloc"), exit(1);

	while (*str)
		if (*str == '{' && str[1] != '{') {
			int l;
			char *file = st->file;
			size_t filec = st->filec;
			char *s;
			st->file = ++str, st->filec = 0;
			/* TODO skip until } to allow for comments */
			if (!(v = read_pair(st, '}')))
				return NULL;
			str += st->filec;
			st->file = file, st->filec = filec;

			if (!(v = eevo_eval_list(st, env, v))) /* TODO sandboxed eval, no mutable procs */
				return NULL;
			if (!(s = eevo_print(v)))
				return NULL;
			/* TODO if last = !d run display converter on it */
			l = strlen(s);
			ret_len += l;

			if (ret_len >= ret_cap) /* resize output if necessary */
				if (!(ret = realloc(ret, ret_cap *= 2)))
					perror("; realloc"), exit(1);
			memcpy(ret + pos, s, l);
			free(s);
			pos += l;
		} else {
			if (*str == '{' || *str == '}') str++; /* only add 1 curly brace when escaped */
			ret[pos++] = *str++;
		}
	ret[pos] = '\0';
	return eevo_str(st, ret);
}

void
eevo_env_string(EevoSt st)
{
	st->types[5]->v.t.func = eevo_prim(st, EEVO_PRIM, prim_Str, "Str");
	st->types[6]->v.t.func = eevo_prim(st, EEVO_PRIM, prim_Sym, "Sym");
	eevo_env_prim(strlen);
	eevo_env_form(strfmt);
}
/* zlib License
 *
 * Copyright (c) 2017-2026 Ed van Bruggen
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */
#include <assert.h>
#include <math.h>

#define EVAL_CHECK(A, V, NAME, TYPE) do {  \
	if (!(A = eevo_eval(st, vars, V))) \
		return NULL;               \
	eevo_arg_type(A, NAME, TYPE);      \
} while(0)

/* wrapper functions to be returned by eevo_num, all need same arguments */
static Eevo
create_int(EevoSt st, double num, double den)
{
	assert(den == 1);
	return eevo_int(st, num);
}

static Eevo
create_dec(EevoSt st, double num, double den)
{
	assert(den == 1);
	return eevo_dec(st, num);
}

static Eevo
create_rat(EevoSt st, double num, double den)
{
	return eevo_rat(st, num, den);
}

/* return pointer to one of the preceding functions depending on what
 * number should be created by the following arithmetic functions
 * force arg is used to force number to one type:
 *   0 -> no force, 1 -> force ratio/int, 2 -> force decimal */
static Eevo
(*eevo_num(EevoType a, EevoType b, int force))(EevoSt, double, double)
{
	if (force == 1)
		return &create_rat;
	if (force == 2)
		return &create_dec;
	if (a & EEVO_DEC || b & EEVO_DEC)
		return &create_dec;
	if (a & EEVO_RATIO || b & EEVO_RATIO)
		return &create_rat;
	return &create_int;
}

#define PRIM_ROUND(NAME, FORCE)                                        \
static Eevo                                                            \
prim_##NAME(EevoSt st, EevoRec vars, Eevo args)                        \
{                                                                      \
	Eevo n;                                                        \
	eevo_arg_num(args, #NAME, 1);                                  \
	n = fst(args);                                                 \
	eevo_arg_type(n, #NAME, EEVO_NUM);                             \
	return (eevo_num(n->t, n->t, FORCE))(st, NAME(num(n)/den(n)), 1.); \
}

/* define int and dec as identity functions to use them in the same macro */
#define Int(X) (X)
#define Dec(X) (X)
PRIM_ROUND(Int,   1)
PRIM_ROUND(Dec,   2)
#undef Int
#undef Dec
PRIM_ROUND(round, 0)
PRIM_ROUND(floor, 0)
PRIM_ROUND(ceil,  0)

static Eevo
prim_add(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo a, b;
	eevo_arg_num(args, "+", 2);
	a = fst(args), b = snd(args);
	eevo_arg_type(a, "+", EEVO_NUM);
	eevo_arg_type(b, "+", EEVO_NUM);
	if (a->t & EEVO_DEC || b->t & EEVO_DEC)
		return eevo_dec(st, (num(a)/den(a)) + (num(b)/den(b)));
	return (eevo_num(a->t, b->t, 0))
		(st, num(a) * den(b) + den(a) * num(b),
		     den(a) * den(b));
}

static Eevo
prim_sub(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo a, b;
	int len = eevo_lstlen(args);
	if (len != 2 && len != 1)
		eevo_warnf("-: expected 1 or 2 arguments, recieved %d", len);
	a = fst(args);
	eevo_arg_type(a, "-", EEVO_NUM);
	if (len == 1) {
		b = a;
		a = eevo_int(st, 0);
	} else {
		b = snd(args);
		eevo_arg_type(b, "-", EEVO_NUM);
	}
	if (a->t & EEVO_DEC || b->t & EEVO_DEC)
		return eevo_dec(st, (num(a)/den(a)) - (num(b)/den(b)));
	return (eevo_num(a->t, b->t, 0))
		(st, num(a) * den(b) - den(a) * num(b),
		     den(a) * den(b));
}

static Eevo
prim_mul(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo a, b;
	eevo_arg_num(args, "*", 2);
	a = fst(args), b = snd(args);
	eevo_arg_type(a, "*", EEVO_NUM);
	eevo_arg_type(b, "*", EEVO_NUM);
	if (a->t & EEVO_DEC || b->t & EEVO_DEC)
		return eevo_dec(st, (num(a)/den(a)) * (num(b)/den(b)));
	return (eevo_num(a->t, b->t, 0))(st, num(a) * num(b), den(a) * den(b));

}

static Eevo
prim_div(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo a, b;
	int len = eevo_lstlen(args);
	if (len != 2 && len != 1)
		eevo_warnf("/: expected 1 or 2 arguments, recieved %d", len);
	a = fst(args);
	eevo_arg_type(a, "/", EEVO_NUM);
	if (len == 1) {
		b = a;
		a = eevo_int(st, 1);
	} else {
		b = snd(args);
		eevo_arg_type(b, "/", EEVO_NUM);
	}
	if (a->t & EEVO_DEC || b->t & EEVO_DEC)
		return eevo_dec(st, (num(a)/den(a)) / (num(b)/den(b)));
	return (eevo_num(a->t, b->t, 1))(st, num(a) * den(b), den(a) * num(b));
}

static Eevo
prim_mod(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo a, b;
	eevo_arg_num(args, "mod", 2);
	a = fst(args), b = snd(args);
	eevo_arg_type(a, "mod", EEVO_INT);
	eevo_arg_type(b, "mod", EEVO_INT);
	if (num(b) == 0)
		eevo_warn("division by zero");
	return eevo_int(st, (int)num(a) % abs((int)num(b)));
}

/* TODO if given function as 2nd arg run it on first arg */
static Eevo
prim_pow(EevoSt st, EevoRec vars, Eevo args)
{
	Eevo b, p;
	double bnum, bden;
	eevo_arg_num(args, "pow", 2);
	b = fst(args), p = snd(args);
	eevo_arg_type(b, "pow", EEVO_EXPR);
	eevo_arg_type(p, "pow", EEVO_EXPR);
	bnum = pow(num(b), num(p)/den(p));
	bden = pow(den(b), num(p)/den(p));
	if ((bnum == (int)bnum && bden == (int)bden) ||
	     b->t & EEVO_DEC || p->t & EEVO_DEC)
		return eevo_num(b->t, p->t, 0)(st, bnum, bden);
	return eevo_list(st, 3, eevo_sym(st, "^"), b, p);
}

#define PRIM_COMPARE(NAME, OP)                      \
static Eevo                                         \
prim_##NAME(EevoSt st, EevoRec vars, Eevo args)     \
{                                                   \
	if (eevo_lstlen(args) != 2)                 \
		return True;                        \
	eevo_arg_type(fst(args), #OP, EEVO_NUM);    \
	eevo_arg_type(snd(args), #OP, EEVO_NUM);    \
	return ((num(fst(args))*den(snd(args)))  OP \
		(num(snd(args))*den(fst(args)))) ?  \
		True : Nil;                         \
}

PRIM_COMPARE(lt,  <)
PRIM_COMPARE(gt,  >)
PRIM_COMPARE(lte, <=)
PRIM_COMPARE(gte, >=)

#define PRIM_TRIG(NAME)                                          \
static Eevo                                                      \
prim_##NAME(EevoSt st, EevoRec vars, Eevo args)                  \
{                                                                \
	eevo_arg_num(args, #NAME, 1);                            \
	eevo_arg_type(fst(args), #NAME, EEVO_EXPR);              \
	if (fst(args)->t & EEVO_DEC)                             \
		return eevo_dec(st, NAME(num(fst(args))));       \
	return eevo_list(st, 2, eevo_sym(st, #NAME), fst(args)); \
}

PRIM_TRIG(sin)
PRIM_TRIG(cos)
PRIM_TRIG(tan)
PRIM_TRIG(sinh)
PRIM_TRIG(cosh)
PRIM_TRIG(tanh)
PRIM_TRIG(asin)
PRIM_TRIG(acos)
PRIM_TRIG(atan)
PRIM_TRIG(asinh)
PRIM_TRIG(acosh)
PRIM_TRIG(atanh)
PRIM_TRIG(exp)
PRIM_TRIG(log)

static Eevo
prim_numerator(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "numerator", 1);
	eevo_arg_type(fst(args), "numerator", EEVO_INT | EEVO_RATIO);
	return eevo_int(st, fst(args)->v.n.num);
}

static Eevo
prim_denominator(EevoSt st, EevoRec env, Eevo args)
{
	eevo_arg_num(args, "denominator", 1);
	eevo_arg_type(fst(args), "denominator", EEVO_INT | EEVO_RATIO);
	return eevo_int(st, fst(args)->v.n.den);
}

void
eevo_env_math(EevoSt st)
{
	st->types[2]->v.t.func = eevo_prim(st, EEVO_PRIM, prim_Int, "Int");
	st->types[3]->v.t.func = eevo_prim(st, EEVO_PRIM, prim_Dec, "Dec");
	eevo_env_prim(floor);
	eevo_env_prim(ceil);
	eevo_env_prim(round);
	eevo_env_prim(numerator);
	eevo_env_prim(denominator);

	eevo_env_name_prim(+, add);
	eevo_env_name_prim(-, sub);
	eevo_env_name_prim(*, mul);
	eevo_env_name_prim(/, div);
	eevo_env_name_prim(^, pow);
	eevo_env_prim(mod);

	eevo_env_name_prim(<,  lt);
	eevo_env_name_prim(>,  gt);
	eevo_env_name_prim(<=, lte);
	eevo_env_name_prim(>=, gte);

	eevo_env_prim(sin);
	eevo_env_prim(cos);
	eevo_env_prim(tan);
	eevo_env_prim(sinh);
	eevo_env_prim(cosh);
	eevo_env_prim(tanh);
	eevo_env_name_prim(arcsin,  asin);
	eevo_env_name_prim(arccos,  acos);
	eevo_env_name_prim(arctan,  atan);
	eevo_env_name_prim(arcsinh, asinh);
	eevo_env_name_prim(arccosh, acosh);
	eevo_env_name_prim(arctanh, atanh);
	eevo_env_prim(exp);
	eevo_env_prim(log);
}
char eevo_core[] = { 
  0x3b, 0x3b, 0x3b, 0x20, 0x63, 0x6f, 0x72, 0x65, 0x2e, 0x65, 0x76, 0x6f,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x28, 0x2e, 0x2e,
  0x2e, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x22, 0x43, 0x72, 0x65, 0x61,
  0x74, 0x65, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x20, 0x6c, 0x73, 0x74,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x71, 0x75, 0x69, 0x74, 0x20, 0x27, 0x7e,
  0x4e, 0x6f, 0x20, 0x52, 0x45, 0x50, 0x4c, 0x20, 0x74, 0x6f, 0x20, 0x71,
  0x75, 0x69, 0x74, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x7e, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x0a,
  0x20, 0x20, 0x4d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28, 0x61, 0x72, 0x67,
  0x73, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x22, 0x44, 0x65, 0x66, 0x69, 0x6e, 0x65, 0x20,
  0x6e, 0x61, 0x6d, 0x65, 0x64, 0x20, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x2c,
  0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x72, 0x67, 0x75, 0x6d, 0x65,
  0x6e, 0x74, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x61, 0x6e, 0x64, 0x20,
  0x62, 0x6f, 0x64, 0x79, 0x0a, 0x20, 0x20, 0x46, 0x69, 0x72, 0x73, 0x74,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20,
  0x61, 0x72, 0x67, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x69, 0x73,
  0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x6d, 0x61, 0x63,
  0x72, 0x6f, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65,
  0x65, 0x3a, 0x20, 0x64, 0x65, 0x66, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x70, 0x61, 0x69, 0x72, 0x3f, 0x20, 0x61, 0x72, 0x67, 0x73, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x20, 0x27, 0x64, 0x65, 0x66, 0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x61,
  0x72, 0x67, 0x73, 0x29, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27,
  0x4d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x61,
  0x72, 0x67, 0x73, 0x29, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64,
  0x79, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c, 0x73,
  0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x72,
  0x72, 0x6f, 0x72, 0x20, 0x27, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72,
  0x6f, 0x20, 0x22, 0x65, 0x78, 0x70, 0x65, 0x63, 0x74, 0x65, 0x64, 0x20,
  0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x61,
  0x6e, 0x64, 0x20, 0x61, 0x72, 0x67, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x20,
  0x4c, 0x69, 0x73, 0x74, 0x2c, 0x20, 0x72, 0x65, 0x63, 0x69, 0x65, 0x76,
  0x65, 0x64, 0x20, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x74, 0x79, 0x70, 0x65, 0x6f,
  0x66, 0x20, 0x61, 0x72, 0x67, 0x73, 0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20,
  0x66, 0x73, 0x74, 0x2f, 0x72, 0x73, 0x74, 0x0a, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x73, 0x6e, 0x64, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x66, 0x73,
  0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x66, 0x66, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x20, 0x66,
  0x73, 0x74, 0x28, 0x66, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x72, 0x66, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x20,
  0x72, 0x73, 0x74, 0x28, 0x66, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x72, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20,
  0x20, 0x72, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x66, 0x66, 0x66, 0x73, 0x74, 0x28, 0x78,
  0x29, 0x20, 0x66, 0x73, 0x74, 0x28, 0x66, 0x73, 0x74, 0x28, 0x66, 0x73,
  0x74, 0x28, 0x78, 0x29, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x66,
  0x66, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x66, 0x73, 0x74, 0x28,
  0x66, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x66, 0x72, 0x66, 0x73, 0x74, 0x28, 0x78,
  0x29, 0x20, 0x66, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x66, 0x73,
  0x74, 0x28, 0x78, 0x29, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x66,
  0x72, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x66, 0x73, 0x74, 0x28,
  0x72, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x66, 0x66, 0x73, 0x74, 0x28, 0x78,
  0x29, 0x20, 0x72, 0x73, 0x74, 0x28, 0x66, 0x73, 0x74, 0x28, 0x66, 0x73,
  0x74, 0x28, 0x78, 0x29, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72,
  0x66, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x72, 0x73, 0x74, 0x28,
  0x66, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x72, 0x66, 0x73, 0x74, 0x28, 0x78,
  0x29, 0x20, 0x72, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x66, 0x73,
  0x74, 0x28, 0x78, 0x29, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72,
  0x72, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x72, 0x73, 0x74, 0x28,
  0x72, 0x73, 0x74, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x29,
  0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x54, 0x79, 0x70, 0x65, 0x73, 0x0a,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x6e, 0x79, 0x3f, 0x28, 0x78, 0x29,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x54, 0x72, 0x75,
  0x65, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x76, 0x6f, 0x69, 0x64, 0x3f, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d,
  0x20, 0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x54, 0x56, 0x6f,
  0x69, 0x64, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6e, 0x69, 0x6c, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x54,
  0x4e, 0x69, 0x6c, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x69, 0x6e, 0x74,
  0x65, 0x67, 0x65, 0x72, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20,
  0x49, 0x6e, 0x74, 0x29, 0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20,
  0x73, 0x68, 0x6f, 0x72, 0x74, 0x65, 0x6e, 0x20, 0x74, 0x79, 0x70, 0x65,
  0x20, 0x71, 0x75, 0x65, 0x72, 0x72, 0x79, 0x20, 0x66, 0x75, 0x6e, 0x63,
  0x73, 0x3a, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x3f, 0x20,
  0x2d, 0x3e, 0x20, 0x69, 0x6e, 0x74, 0x3f, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x64, 0x65, 0x63, 0x69, 0x6d, 0x61, 0x6c, 0x3f, 0x28, 0x78, 0x29, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28,
  0x78, 0x29, 0x20, 0x44, 0x65, 0x63, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x72, 0x61, 0x74, 0x69, 0x6f, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28,
  0x78, 0x29, 0x20, 0x52, 0x61, 0x74, 0x69, 0x6f, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x3f, 0x28, 0x78, 0x29,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70,
  0x65, 0x28, 0x78, 0x29, 0x20, 0x53, 0x74, 0x72, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x73, 0x79, 0x6d, 0x62, 0x6f, 0x6c, 0x3f, 0x28, 0x78, 0x29,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70,
  0x65, 0x28, 0x78, 0x29, 0x20, 0x53, 0x79, 0x6d, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x70, 0x72, 0x69, 0x6d, 0x69, 0x74, 0x69, 0x76, 0x65, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79, 0x70,
  0x65, 0x28, 0x78, 0x29, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x29, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x66, 0x6f,
  0x72, 0x6d, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x28, 0x3d, 0x20, 0x54, 0x79,
  0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x46, 0x6f, 0x72, 0x6d, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e,
  0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x54,
  0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x46, 0x75, 0x6e, 0x63, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x3f, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20,
  0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x4d, 0x61, 0x63, 0x72,
  0x6f, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x72,
  0x64, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x52, 0x65,
  0x63, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x3d, 0x20, 0x54, 0x79, 0x70, 0x65, 0x28, 0x78, 0x29, 0x20, 0x50, 0x61,
  0x69, 0x72, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x74, 0x6f, 0x6d,
  0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x6e, 0x6f, 0x74, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x78,
  0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x69, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x78, 0x29, 0x20,
  0x6c, 0x69, 0x73, 0x74, 0x3f, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29,
  0x29, 0x20, 0x6e, 0x6f, 0x74, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x62, 0x6f, 0x6f, 0x6c, 0x65, 0x61, 0x6e, 0x3f, 0x28, 0x78,
  0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x28, 0x3d,
  0x20, 0x78, 0x20, 0x54, 0x72, 0x75, 0x65, 0x29, 0x20, 0x6e, 0x69, 0x6c,
  0x3f, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x74, 0x72,
  0x75, 0x65, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x3d, 0x20, 0x78, 0x20, 0x54, 0x72, 0x75, 0x65, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x66, 0x61, 0x6c, 0x73, 0x65, 0x3f, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x69, 0x6c,
  0x3f, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x62, 0x75, 0x69, 0x6c, 0x74, 0x69,
  0x6e, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f,
  0x72, 0x20, 0x70, 0x72, 0x69, 0x6d, 0x69, 0x74, 0x69, 0x76, 0x65, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x73, 0x70, 0x65, 0x63, 0x69, 0x61, 0x6c, 0x66,
  0x6f, 0x72, 0x6d, 0x3f, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x64, 0x75, 0x72, 0x65, 0x3f, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x62, 0x75, 0x69,
  0x6c, 0x74, 0x69, 0x6e, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x28, 0x6f, 0x72,
  0x20, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x3f, 0x28, 0x78,
  0x29, 0x20, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x3f, 0x28, 0x78, 0x29, 0x29,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e,
  0x61, 0x6c, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f,
  0x72, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x3f, 0x28, 0x78,
  0x29, 0x20, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x3f, 0x28, 0x78, 0x29, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f, 0x72,
  0x20, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x3f, 0x28, 0x78,
  0x29, 0x20, 0x64, 0x65, 0x63, 0x69, 0x6d, 0x61, 0x6c, 0x3f, 0x28, 0x78,
  0x29, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x42, 0x6f, 0x6f, 0x6c,
  0x28, 0x78, 0x29, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x78, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x54, 0x72, 0x75, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x4e, 0x69, 0x6c, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72,
  0x6f, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x28, 0x65, 0x78, 0x70,
  0x72, 0x29, 0x0a, 0x20, 0x20, 0x60, 0x28, 0x75, 0x6e, 0x6c, 0x65, 0x73,
  0x73, 0x20, 0x2c, 0x65, 0x78, 0x70, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x61, 0x73, 0x73,
  0x65, 0x72, 0x74, 0x20, 0x22, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x69,
  0x6f, 0x6e, 0x20, 0x22, 0x20, 0x27, 0x2c, 0x65, 0x78, 0x70, 0x72, 0x20,
  0x22, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x65, 0x64, 0x22, 0x29, 0x29, 0x0a,
  0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c,
  0x20, 0x46, 0x6c, 0x6f, 0x77, 0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44,
  0x4f, 0x20, 0x69, 0x66, 0x20, 0x62, 0x20, 0x3d, 0x20, 0x70, 0x61, 0x69,
  0x72, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x66, 0x73, 0x74, 0x20, 0x62, 0x20,
  0x3d, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x75, 0x73, 0x65, 0x20, 0x72,
  0x73, 0x74, 0x20, 0x62, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72,
  0x6f, 0x20, 0x28, 0x69, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x20, 0x61, 0x20,
  0x62, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x78, 0x65, 0x63, 0x75, 0x74,
  0x65, 0x20, 0x61, 0x20, 0x69, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69,
  0x74, 0x69, 0x6f, 0x6e, 0x20, 0x63, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20,
  0x74, 0x72, 0x75, 0x65, 0x2c, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x77,
  0x69, 0x73, 0x65, 0x20, 0x72, 0x75, 0x6e, 0x20, 0x62, 0x22, 0x0a, 0x20,
  0x20, 0x5b, 0x63, 0x6f, 0x6e, 0x64, 0x20, 0x5b, 0x63, 0x6f, 0x6e, 0x20,
  0x61, 0x5d, 0x20, 0x5b, 0x54, 0x72, 0x75, 0x65, 0x20, 0x62, 0x5d, 0x5d,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x54, 0x72,
  0x75, 0x65, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20,
  0x28, 0x77, 0x68, 0x65, 0x6e, 0x20, 0x63, 0x6f, 0x6e, 0x20, 0x2e, 0x2e,
  0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x45,
  0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20,
  0x69, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e,
  0x20, 0x63, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x74, 0x72, 0x75, 0x65,
  0x22, 0x0a, 0x20, 0x20, 0x5b, 0x63, 0x6f, 0x6e, 0x64, 0x20, 0x5b, 0x63,
  0x6f, 0x6e, 0x20, 0x5b, 0x64, 0x6f, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62,
  0x6f, 0x64, 0x79, 0x5d, 0x5d, 0x5d, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61,
  0x63, 0x72, 0x6f, 0x20, 0x28, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20,
  0x63, 0x6f, 0x6e, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65,
  0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73,
  0x20, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x20,
  0x63, 0x6f, 0x6e, 0x2c, 0x20, 0x69, 0x73, 0x20, 0x74, 0x72, 0x75, 0x65,
  0x22, 0x0a, 0x20, 0x20, 0x5b, 0x63, 0x6f, 0x6e, 0x64, 0x20, 0x5b, 0x5b,
  0x6e, 0x6f, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x5d, 0x20, 0x5b, 0x64, 0x6f,
  0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x5d, 0x5d, 0x5d,
  0x0a, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28,
  0x6c, 0x65, 0x74, 0x20, 0x76, 0x61, 0x72, 0x73, 0x20, 0x2e, 0x2e, 0x2e,
  0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x78,
  0x65, 0x63, 0x75, 0x74, 0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x77,
  0x69, 0x74, 0x68, 0x20, 0x6e, 0x65, 0x77, 0x20, 0x6c, 0x6f, 0x63, 0x61,
  0x6c, 0x20, 0x76, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x73, 0x20,
  0x69, 0x6e, 0x20, 0x76, 0x61, 0x72, 0x73, 0x0a, 0x20, 0x20, 0x76, 0x61,
  0x72, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x20, 0x6f, 0x66, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x61, 0x6e, 0x64,
  0x20, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x20, 0x70, 0x61, 0x69, 0x72, 0x73,
  0x22, 0x0a, 0x20, 0x20, 0x5b, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20,
  0x27, 0x46, 0x75, 0x6e, 0x63, 0x20, 0x28, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x28,
  0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x27, 0x64, 0x65, 0x66, 0x20, 0x28,
  0x66, 0x73, 0x74, 0x20, 0x69, 0x74, 0x29, 0x20, 0x28, 0x72, 0x73, 0x74,
  0x20, 0x69, 0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x76, 0x61, 0x72, 0x73, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x29,
  0x5d, 0x0a, 0x0a, 0x28, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f,
  0x20, 0x28, 0x72, 0x65, 0x63, 0x75, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x63,
  0x20, 0x76, 0x61, 0x72, 0x73, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f,
  0x64, 0x79, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x44, 0x6f, 0x20, 0x72, 0x65,
  0x63, 0x75, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x77, 0x69, 0x74, 0x68,
  0x69, 0x6e, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x62, 0x79, 0x20, 0x63,
  0x61, 0x6c, 0x6c, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20,
  0x77, 0x69, 0x74, 0x68, 0x20, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x73, 0x20,
  0x66, 0x6f, 0x72, 0x20, 0x76, 0x61, 0x72, 0x73, 0x0a, 0x20, 0x20, 0x41,
  0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x6c, 0x65, 0x74,
  0x22, 0x0a, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27, 0x6c,
  0x65, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x6c, 0x69, 0x73, 0x74, 0x2a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x70, 0x72,
  0x6f, 0x63, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x27, 0x46,
  0x75, 0x6e, 0x63, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x20, 0x66, 0x73, 0x74,
  0x20, 0x76, 0x61, 0x72, 0x73, 0x29, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x76, 0x61, 0x72, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x70, 0x72, 0x6f,
  0x63, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x20, 0x66, 0x73, 0x74, 0x20, 0x76,
  0x61, 0x72, 0x73, 0x29, 0x29, 0x29, 0x29, 0x0a, 0x0a, 0x3b, 0x20, 0x54,
  0x4f, 0x44, 0x4f, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x20,
  0x65, 0x6c, 0x73, 0x65, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x69, 0x73,
  0x20, 0x72, 0x75, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x6e, 0x6f, 0x20, 0x76,
  0x61, 0x6c, 0x75, 0x65, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x65, 0x71,
  0x75, 0x61, 0x6c, 0x0a, 0x28, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72,
  0x6f, 0x20, 0x28, 0x73, 0x77, 0x69, 0x74, 0x63, 0x68, 0x20, 0x76, 0x61,
  0x6c, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x0a,
  0x20, 0x20, 0x22, 0x43, 0x6f, 0x6d, 0x70, 0x61, 0x72, 0x65, 0x20, 0x76,
  0x61, 0x6c, 0x75, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x66, 0x69, 0x72, 0x73,
  0x74, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e,
  0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x73,
  0x74, 0x61, 0x74, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x6f, 0x6e,
  0x6c, 0x79, 0x20, 0x72, 0x75, 0x6e, 0x6e, 0x69, 0x6e, 0x67, 0x20, 0x6c,
  0x69, 0x6e, 0x65, 0x20, 0x77, 0x68, 0x65, 0x72, 0x65, 0x20, 0x74, 0x68,
  0x65, 0x79, 0x20, 0x61, 0x72, 0x65, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
  0x22, 0x0a, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x27,
  0x63, 0x6f, 0x6e, 0x64, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x40, 0x60, 0x28, 0x28, 0x3d, 0x20, 0x2c, 0x76, 0x61,
  0x6c, 0x20, 0x2c, 0x28, 0x66, 0x73, 0x74, 0x20, 0x69, 0x74, 0x29, 0x29,
  0x20, 0x2c, 0x28, 0x73, 0x6e, 0x64, 0x20, 0x69, 0x74, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x29, 0x29, 0x29,
  0x0a, 0x0a, 0x28, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20,
  0x28, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20,
  0x65, 0x78, 0x70, 0x72, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x63,
  0x75, 0x72, 0x73, 0x69, 0x76, 0x65, 0x6c, 0x79, 0x20, 0x71, 0x75, 0x6f,
  0x74, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x67, 0x69, 0x76, 0x65, 0x6e,
  0x20, 0x65, 0x78, 0x70, 0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x0a,
  0x20, 0x20, 0x41, 0x75, 0x74, 0x6f, 0x6d, 0x61, 0x74, 0x69, 0x63, 0x61,
  0x6c, 0x6c, 0x79, 0x20, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x73, 0x20, 0x65,
  0x61, 0x63, 0x68, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20,
  0x77, 0x69, 0x74, 0x68, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x65,
  0x78, 0x70, 0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x62,
  0x75, 0x74, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74, 0x65, 0x73,
  0x20, 0x74, 0x68, 0x65, 0x0a, 0x20, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65,
  0x6e, 0x74, 0x20, 0x69, 0x66, 0x20, 0x69, 0x74, 0x20, 0x69, 0x73, 0x20,
  0x6c, 0x61, 0x62, 0x65, 0x6c, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
  0x20, 0x74, 0x68, 0x65, 0x20, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x20, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x2e, 0x0a, 0x20, 0x20, 0x43, 0x61,
  0x6e, 0x20, 0x62, 0x65, 0x20, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x65, 0x6e,
  0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20,
  0x60, 0x20, 0x70, 0x72, 0x65, 0x66, 0x69, 0x78, 0x2e, 0x0a, 0x20, 0x20,
  0x41, 0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x71, 0x75,
  0x6f, 0x74, 0x65, 0x2c, 0x20, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x2c, 0x20, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73, 0x70,
  0x6c, 0x69, 0x63, 0x65, 0x22, 0x0a, 0x20, 0x20, 0x28, 0x64, 0x65, 0x66,
  0x20, 0x28, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x20, 0x66, 0x6f, 0x72, 0x6d,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x75, 0x6e, 0x6c, 0x65, 0x73,
  0x73, 0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x28, 0x70, 0x61, 0x69, 0x72,
  0x3f, 0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x29,
  0x29, 0x20, 0x28, 0x6e, 0x69, 0x6c, 0x3f, 0x20, 0x28, 0x72, 0x72, 0x73,
  0x74, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x28,
  0x66, 0x73, 0x74, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x29, 0x20, 0x22, 0x69,
  0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x20,
  0x22, 0x20, 0x66, 0x6f, 0x72, 0x6d, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x28, 0x64, 0x65, 0x66, 0x20, 0x28, 0x71, 0x75, 0x61, 0x73, 0x69, 0x63,
  0x6f, 0x6e, 0x73, 0x20, 0x61, 0x20, 0x64, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x69, 0x66, 0x20, 0x28, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x20,
  0x64, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66,
  0x20, 0x28, 0x3d, 0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x64, 0x29, 0x20,
  0x27, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20, 0x28, 0x61, 0x6e, 0x64,
  0x20, 0x28, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x20, 0x61, 0x29, 0x20, 0x28,
  0x3d, 0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x61, 0x29, 0x20, 0x27, 0x71,
  0x75, 0x6f, 0x74, 0x65, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27,
  0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a,
  0x20, 0x28, 0x73, 0x6e, 0x64, 0x20, 0x61, 0x29, 0x20, 0x28, 0x73, 0x6e,
  0x64, 0x20, 0x64, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20, 0x28, 0x6e, 0x69,
  0x6c, 0x3f, 0x20, 0x28, 0x73, 0x6e, 0x64, 0x20, 0x64, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27, 0x6c, 0x69, 0x73, 0x74, 0x20,
  0x61, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6c, 0x69, 0x73,
  0x74, 0x2a, 0x20, 0x61, 0x20, 0x64, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20, 0x28, 0x6d,
  0x65, 0x6d, 0x62, 0x65, 0x72, 0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x64,
  0x29, 0x20, 0x27, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6c, 0x69, 0x73,
  0x74, 0x2a, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x28, 0x66,
  0x73, 0x74, 0x20, 0x64, 0x29, 0x20, 0x61, 0x20, 0x28, 0x72, 0x73, 0x74,
  0x20, 0x64, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6c, 0x69, 0x73,
  0x74, 0x2a, 0x20, 0x61, 0x20, 0x64, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6c, 0x69,
  0x73, 0x74, 0x2a, 0x20, 0x61, 0x20, 0x64, 0x29, 0x29, 0x29, 0x0a, 0x20,
  0x20, 0x28, 0x72, 0x65, 0x63, 0x75, 0x72, 0x20, 0x66, 0x20, 0x28, 0x28,
  0x65, 0x78, 0x70, 0x72, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x28,
  0x6e, 0x20, 0x30, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63,
  0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x28,
  0x6e, 0x69, 0x6c, 0x3f, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x4e,
  0x69, 0x6c, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x28,
  0x61, 0x74, 0x6f, 0x6d, 0x3f, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20,
  0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x28, 0x3d, 0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x65,
  0x78, 0x70, 0x72, 0x29, 0x20, 0x27, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71,
  0x75, 0x6f, 0x74, 0x65, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x20, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x71, 0x75,
  0x61, 0x73, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x20, 0x27, 0x27, 0x71, 0x75,
  0x61, 0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x28, 0x66, 0x20,
  0x28, 0x72, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x28,
  0x2b, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x28, 0x3d, 0x20, 0x28, 0x66, 0x73, 0x74,
  0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x27, 0x75, 0x6e, 0x71, 0x75,
  0x6f, 0x74, 0x65, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29,
  0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20,
  0x28, 0x3d, 0x20, 0x6e, 0x20, 0x30, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x73, 0x6e, 0x64, 0x20, 0x65, 0x78,
  0x70, 0x72, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x71, 0x75, 0x61, 0x73, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x20,
  0x27, 0x27, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x28, 0x66,
  0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20,
  0x28, 0x2d, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x29, 0x29, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x28, 0x3d, 0x20, 0x28, 0x66,
  0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x27, 0x75, 0x6e,
  0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73, 0x70, 0x6c, 0x69, 0x63, 0x65,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x68,
  0x65, 0x63, 0x6b, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20, 0x28, 0x3d, 0x20,
  0x6e, 0x20, 0x30, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x75, 0x6e,
  0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73, 0x70, 0x6c, 0x69, 0x63, 0x65,
  0x20, 0x22, 0x69, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x63, 0x6f,
  0x6e, 0x74, 0x65, 0x78, 0x74, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x22, 0x20,
  0x28, 0x73, 0x6e, 0x64, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x71, 0x75,
  0x61, 0x73, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x20, 0x27, 0x27, 0x75, 0x6e,
  0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73, 0x70, 0x6c, 0x69, 0x63, 0x65,
  0x20, 0x28, 0x66, 0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70,
  0x72, 0x29, 0x20, 0x28, 0x2d, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x29, 0x29,
  0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x28, 0x61,
  0x6e, 0x64, 0x20, 0x28, 0x3d, 0x20, 0x6e, 0x20, 0x30, 0x29, 0x20, 0x28,
  0x61, 0x6e, 0x64, 0x20, 0x28, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x20, 0x28,
  0x66, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x29, 0x20, 0x28,
  0x3d, 0x20, 0x28, 0x66, 0x66, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x20, 0x27, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73,
  0x70, 0x6c, 0x69, 0x63, 0x65, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x20, 0x28,
  0x66, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6c, 0x65, 0x74, 0x20, 0x28,
  0x64, 0x3a, 0x20, 0x28, 0x66, 0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x65,
  0x78, 0x70, 0x72, 0x29, 0x20, 0x6e, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x69, 0x66, 0x20, 0x64, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x6c, 0x69, 0x73, 0x74, 0x20, 0x27, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64,
  0x20, 0x28, 0x66, 0x72, 0x66, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x20, 0x64, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x66, 0x72, 0x66, 0x73, 0x74, 0x20, 0x65,
  0x78, 0x70, 0x72, 0x29, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x28, 0x71, 0x75, 0x61,
  0x73, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x20, 0x28, 0x66, 0x20, 0x28, 0x66,
  0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x20, 0x6e, 0x29, 0x20,
  0x28, 0x66, 0x20, 0x28, 0x72, 0x73, 0x74, 0x20, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x20, 0x6e, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x0a, 0x0a, 0x28,
  0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28, 0x75, 0x6e,
  0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x65, 0x78, 0x70, 0x72, 0x29, 0x0a,
  0x20, 0x20, 0x22, 0x55, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x65,
  0x78, 0x70, 0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x73, 0x6f,
  0x20, 0x69, 0x74, 0x73, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74,
  0x65, 0x64, 0x20, 0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x70, 0x6c,
  0x61, 0x63, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x74, 0x6f, 0x20, 0x74, 0x68,
  0x65, 0x20, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x0a, 0x20, 0x20, 0x43, 0x61, 0x6e, 0x20, 0x62, 0x65, 0x20, 0x73, 0x68,
  0x6f, 0x72, 0x74, 0x65, 0x6e, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
  0x20, 0x74, 0x68, 0x65, 0x20, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x66, 0x69,
  0x78, 0x0a, 0x20, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x73, 0x20, 0x69,
  0x66, 0x20, 0x63, 0x61, 0x6c, 0x6c, 0x65, 0x64, 0x20, 0x6f, 0x75, 0x74,
  0x73, 0x69, 0x64, 0x65, 0x20, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75,
  0x6f, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f, 0x20, 0x73,
  0x65, 0x65, 0x3a, 0x20, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f,
  0x74, 0x65, 0x2c, 0x20, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d,
  0x73, 0x70, 0x6c, 0x69, 0x63, 0x65, 0x2c, 0x20, 0x71, 0x75, 0x6f, 0x74,
  0x65, 0x22, 0x0a, 0x20, 0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20,
  0x27, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x22, 0x63, 0x61,
  0x6c, 0x6c, 0x65, 0x64, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x69, 0x64, 0x65,
  0x20, 0x6f, 0x66, 0x20, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f,
  0x74, 0x65, 0x22, 0x29, 0x29, 0x0a, 0x28, 0x64, 0x65, 0x66, 0x6d, 0x61,
  0x63, 0x72, 0x6f, 0x20, 0x28, 0x75, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x2d, 0x73, 0x70, 0x6c, 0x69, 0x63, 0x65, 0x20, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x55, 0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65,
  0x20, 0x61, 0x6e, 0x64, 0x20, 0x73, 0x70, 0x6c, 0x69, 0x63, 0x65, 0x20,
  0x74, 0x68, 0x65, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65, 0x73, 0x73, 0x69,
  0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20,
  0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x0a, 0x20,
  0x20, 0x49, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x76, 0x61, 0x6c, 0x75,
  0x65, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74, 0x65, 0x64, 0x20,
  0x69, 0x73, 0x20, 0x61, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x2c, 0x20, 0x65,
  0x6d, 0x62, 0x65, 0x64, 0x64, 0x65, 0x64, 0x20, 0x65, 0x61, 0x63, 0x68,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x74,
  0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x71, 0x75, 0x61, 0x73, 0x69, 0x71,
  0x75, 0x6f, 0x74, 0x65, 0x0a, 0x20, 0x20, 0x43, 0x61, 0x6e, 0x20, 0x62,
  0x65, 0x20, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x65, 0x6e, 0x65, 0x64, 0x20,
  0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68, 0x65, 0x20, 0x2c, 0x40, 0x20,
  0x70, 0x72, 0x65, 0x66, 0x69, 0x78, 0x0a, 0x20, 0x20, 0x45, 0x72, 0x72,
  0x6f, 0x72, 0x73, 0x20, 0x69, 0x66, 0x20, 0x63, 0x61, 0x6c, 0x6c, 0x65,
  0x64, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x69, 0x64, 0x65, 0x20, 0x61, 0x20,
  0x71, 0x75, 0x61, 0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x20, 0x6f,
  0x66, 0x20, 0x61, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x41,
  0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x71, 0x75, 0x61,
  0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x2c, 0x20, 0x75, 0x6e, 0x71,
  0x75, 0x6f, 0x74, 0x65, 0x2c, 0x20, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x22,
  0x0a, 0x20, 0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x75,
  0x6e, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x2d, 0x73, 0x70, 0x6c, 0x69, 0x63,
  0x65, 0x20, 0x22, 0x63, 0x61, 0x6c, 0x6c, 0x65, 0x64, 0x20, 0x6f, 0x75,
  0x74, 0x73, 0x69, 0x64, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x71, 0x75, 0x61,
  0x73, 0x69, 0x71, 0x75, 0x6f, 0x74, 0x65, 0x22, 0x29, 0x29, 0x0a, 0x0a,
  0x3b, 0x3b, 0x3b, 0x20, 0x4c, 0x6f, 0x67, 0x69, 0x63, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x46, 0x61, 0x6c, 0x73, 0x65, 0x20, 0x4e, 0x69, 0x6c,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x28, 0x6e, 0x6f, 0x74, 0x20, 0x78, 0x29,
  0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x78, 0x20, 0x4e, 0x69, 0x6c, 0x20,
  0x54, 0x72, 0x75, 0x65, 0x0a, 0x3b, 0x20, 0x55, 0x73, 0x65, 0x20, 0x61,
  0x20, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x73, 0x6f, 0x20, 0x61, 0x72,
  0x67, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x61, 0x72, 0x65, 0x6e,
  0x27, 0x74, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74, 0x65, 0x64,
  0x20, 0x61, 0x6c, 0x6c, 0x20, 0x61, 0x74, 0x20, 0x6f, 0x6e, 0x63, 0x65,
  0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28, 0x61,
  0x6e, 0x64, 0x20, 0x61, 0x20, 0x62, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52,
  0x65, 0x74, 0x75, 0x72, 0x6e, 0x20, 0x62, 0x20, 0x69, 0x66, 0x20, 0x61,
  0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6e, 0x69, 0x6c, 0x2c,
  0x20, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x72, 0x65, 0x74, 0x75, 0x72, 0x6e,
  0x20, 0x6e, 0x69, 0x6c, 0x22, 0x0a, 0x20, 0x20, 0x5b, 0x69, 0x66, 0x20,
  0x61, 0x20, 0x62, 0x20, 0x4e, 0x69, 0x6c, 0x5d, 0x0a, 0x64, 0x65, 0x66,
  0x6d, 0x61, 0x63, 0x72, 0x6f, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x61, 0x20,
  0x62, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e,
  0x20, 0x61, 0x20, 0x69, 0x66, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x6e, 0x69,
  0x6c, 0x2c, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x72, 0x65, 0x74, 0x75,
  0x72, 0x6e, 0x20, 0x62, 0x22, 0x0a, 0x20, 0x20, 0x5b, 0x69, 0x66, 0x20,
  0x61, 0x20, 0x61, 0x20, 0x62, 0x5d, 0x0a, 0x64, 0x65, 0x66, 0x6d, 0x61,
  0x63, 0x72, 0x6f, 0x20, 0x28, 0x78, 0x6f, 0x72, 0x3f, 0x20, 0x61, 0x20,
  0x62, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x78, 0x63, 0x6c, 0x75, 0x73,
  0x69, 0x76, 0x65, 0x20, 0x6f, 0x72, 0x2c, 0x20, 0x65, 0x69, 0x74, 0x68,
  0x65, 0x72, 0x20, 0x61, 0x20, 0x6f, 0x72, 0x20, 0x62, 0x20, 0x61, 0x72,
  0x65, 0x20, 0x74, 0x72, 0x75, 0x65, 0x2c, 0x20, 0x62, 0x75, 0x74, 0x20,
  0x6e, 0x6f, 0x74, 0x20, 0x69, 0x66, 0x20, 0x62, 0x6f, 0x74, 0x68, 0x20,
  0x61, 0x72, 0x65, 0x20, 0x74, 0x72, 0x75, 0x65, 0x22, 0x0a, 0x20, 0x20,
  0x5b, 0x61, 0x6e, 0x64, 0x20, 0x5b, 0x6f, 0x72, 0x20, 0x61, 0x20, 0x62,
  0x5d, 0x20, 0x5b, 0x6e, 0x6f, 0x74, 0x20, 0x5b, 0x61, 0x6e, 0x64, 0x20,
  0x61, 0x20, 0x62, 0x5d, 0x5d, 0x5d, 0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20,
  0x43, 0x6f, 0x6d, 0x70, 0x6f, 0x73, 0x65, 0x0a, 0x0a, 0x3b, 0x20, 0x54,
  0x4f, 0x44, 0x4f, 0x20, 0x72, 0x65, 0x77, 0x72, 0x69, 0x74, 0x65, 0x2c,
  0x20, 0x6f, 0x70, 0x74, 0x69, 0x6d, 0x69, 0x7a, 0x65, 0x20, 0x74, 0x63,
  0x6f, 0x3f, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f,
  0x73, 0x65, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x73,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x43, 0x72, 0x65, 0x61, 0x74, 0x65, 0x20,
  0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6d, 0x61, 0x64,
  0x65, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x63, 0x68, 0x61, 0x69, 0x6e,
  0x69, 0x6e, 0x67, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x64, 0x75, 0x72,
  0x65, 0x73, 0x20, 0x67, 0x69, 0x76, 0x65, 0x6e, 0x22, 0x0a, 0x20, 0x20,
  0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x69, 0x6c,
  0x3f, 0x28, 0x70, 0x72, 0x6f, 0x63, 0x73, 0x29, 0x3a, 0x20, 0x46, 0x75,
  0x6e, 0x63, 0x28, 0x78, 0x20, 0x78, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x72, 0x73, 0x74, 0x28, 0x70, 0x72, 0x6f,
  0x63, 0x73, 0x29, 0x29, 0x3a, 0x20, 0x66, 0x73, 0x74, 0x28, 0x70, 0x72,
  0x6f, 0x63, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c, 0x73,
  0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x46, 0x75, 0x6e, 0x63,
  0x20, 0x78, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66,
  0x73, 0x74, 0x28, 0x70, 0x72, 0x6f, 0x63, 0x73, 0x29, 0x20, 0x61, 0x70,
  0x70, 0x6c, 0x79, 0x28, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x28, 0x63, 0x6f,
  0x6d, 0x70, 0x6f, 0x73, 0x65, 0x20, 0x72, 0x73, 0x74, 0x28, 0x70, 0x72,
  0x6f, 0x63, 0x73, 0x29, 0x29, 0x20, 0x78, 0x29, 0x0a, 0x0a, 0x3b, 0x3b,
  0x3b, 0x20, 0x52, 0x65, 0x63, 0x6f, 0x72, 0x64, 0x73, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x72, 0x65, 0x63, 0x6b, 0x65, 0x79, 0x73, 0x28, 0x72,
  0x65, 0x63, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x73, 0x74, 0x20,
  0x6f, 0x66, 0x20, 0x6b, 0x65, 0x79, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x72,
  0x65, 0x63, 0x6f, 0x72, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x6d, 0x61, 0x70,
  0x20, 0x66, 0x73, 0x74, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x72, 0x64, 0x73,
  0x28, 0x72, 0x65, 0x63, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x65,
  0x63, 0x76, 0x61, 0x6c, 0x73, 0x28, 0x72, 0x65, 0x63, 0x29, 0x0a, 0x20,
  0x20, 0x22, 0x4c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x76, 0x61,
  0x6c, 0x75, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x6f,
  0x72, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x6d, 0x61, 0x70, 0x20, 0x72, 0x73,
  0x74, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x72, 0x64, 0x73, 0x28, 0x72, 0x65,
  0x63, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x76, 0x6f, 0x63, 0x61,
  0x62, 0x28, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x73, 0x74, 0x20,
  0x6f, 0x66, 0x20, 0x73, 0x79, 0x6d, 0x62, 0x6f, 0x6c, 0x73, 0x20, 0x69,
  0x6e, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x65, 0x6e,
  0x76, 0x69, 0x72, 0x6f, 0x6e, 0x6d, 0x65, 0x6e, 0x74, 0x22, 0x0a, 0x20,
  0x20, 0x28, 0x72, 0x65, 0x63, 0x6b, 0x65, 0x79, 0x73, 0x20, 0x45, 0x4e,
  0x56, 0x29, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x4c, 0x69, 0x73, 0x74, 0x73,
  0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x72, 0x65, 0x6e,
  0x61, 0x6d, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x65, 0x6d, 0x6f, 0x76,
  0x65, 0x20, 0x2a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x2a, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20,
  0x20, 0x22, 0x43, 0x72, 0x65, 0x61, 0x74, 0x65, 0x20, 0x69, 0x6d, 0x70,
  0x72, 0x6f, 0x70, 0x65, 0x72, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x2c, 0x20,
  0x6c, 0x61, 0x73, 0x74, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74,
  0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x4e, 0x69, 0x6c, 0x22,
  0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73,
  0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x20,
  0x66, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x61, 0x70, 0x70,
  0x6c, 0x79, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x2a, 0x20, 0x72, 0x73, 0x74,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x66,
  0x73, 0x74, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x64, 0x6f, 0x30, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x62, 0x6f, 0x64, 0x79,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74,
  0x65, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x65, 0x78, 0x70, 0x72, 0x65,
  0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x6f, 0x64,
  0x79, 0x2c, 0x20, 0x72, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x69, 0x6e, 0x67,
  0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73,
  0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x64, 0x6f, 0x22, 0x0a, 0x20,
  0x20, 0x66, 0x73, 0x74, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x6c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x28, 0x6c, 0x73,
  0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72,
  0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73,
  0x20, 0x69, 0x6e, 0x20, 0x67, 0x69, 0x76, 0x65, 0x6e, 0x20, 0x6c, 0x69,
  0x73, 0x74, 0x22, 0x0a, 0x20, 0x20, 0x72, 0x65, 0x63, 0x75, 0x72, 0x20,
  0x66, 0x20, 0x28, 0x6c, 0x73, 0x74, 0x3a, 0x20, 0x6c, 0x73, 0x74, 0x20,
  0x20, 0x78, 0x3a, 0x20, 0x30, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x69,
  0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29,
  0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x20, 0x72, 0x73, 0x74,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x28, 0x2b, 0x20, 0x78, 0x20, 0x31,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x78, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29,
  0x0a, 0x20, 0x20, 0x22, 0x4c, 0x61, 0x73, 0x74, 0x20, 0x65, 0x6c, 0x65,
  0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x22, 0x0a, 0x20, 0x20, 0x3b, 0x20, 0x72, 0x65, 0x63, 0x75, 0x72, 0x20,
  0x6c, 0x6f, 0x6f, 0x70, 0x20, 0x28, 0x28, 0x6c, 0x73, 0x74, 0x20, 0x6c,
  0x73, 0x74, 0x29, 0x20, 0x28, 0x6e, 0x20, 0x28, 0x69, 0x66, 0x20, 0x6e,
  0x20, 0x28, 0x66, 0x73, 0x74, 0x20, 0x6e, 0x29, 0x20, 0x30, 0x29, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c,
  0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x6c, 0x61, 0x73, 0x74,
  0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x66, 0x73, 0x74, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x0a, 0x3b,
  0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x6d, 0x61, 0x6b, 0x65, 0x20, 0x6e,
  0x74, 0x68, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x69, 0x63, 0x20, 0x66,
  0x6f, 0x72, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x73, 0x74, 0x72, 0x20,
  0x76, 0x65, 0x63, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6e, 0x74, 0x68, 0x28,
  0x6c, 0x73, 0x74, 0x20, 0x6e, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x45, 0x6c,
  0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72,
  0x20, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x2c, 0x20,
  0x73, 0x74, 0x61, 0x72, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72, 0x6f,
  0x6d, 0x20, 0x30, 0x0a, 0x20, 0x20, 0x49, 0x66, 0x20, 0x6e, 0x65, 0x67,
  0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x67, 0x65, 0x74, 0x20, 0x6e, 0x75,
  0x6d, 0x62, 0x65, 0x72, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x65, 0x6e,
  0x64, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a, 0x20,
  0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61, 0x74,
  0x6f, 0x6d, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x6e, 0x74,
  0x68, 0x20, 0x22, 0x69, 0x6e, 0x64, 0x65, 0x78, 0x20, 0x6f, 0x66, 0x20,
  0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f, 0x66, 0x20,
  0x62, 0x6f, 0x75, 0x6e, 0x64, 0x73, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x3c, 0x20, 0x6e, 0x20, 0x30, 0x29, 0x3a, 0x20, 0x6e, 0x74, 0x68,
  0x28, 0x6c, 0x73, 0x74, 0x20, 0x28, 0x2b, 0x20, 0x6c, 0x65, 0x6e, 0x67,
  0x74, 0x68, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x6e, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x6e, 0x20, 0x30, 0x29, 0x3a,
  0x20, 0x66, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20, 0x6e, 0x74, 0x68, 0x28,
  0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x28, 0x2d, 0x20,
  0x6e, 0x20, 0x31, 0x29, 0x29, 0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44,
  0x4f, 0x20, 0x64, 0x69, 0x66, 0x66, 0x20, 0x6e, 0x61, 0x6d, 0x65, 0x20,
  0x68, 0x65, 0x61, 0x64, 0x2f, 0x74, 0x61, 0x69, 0x6c, 0x20, 0x73, 0x69,
  0x6e, 0x63, 0x65, 0x20, 0x63, 0x6f, 0x6e, 0x66, 0x6c, 0x69, 0x63, 0x74,
  0x73, 0x20, 0x77, 0x2f, 0x20, 0x75, 0x6e, 0x69, 0x78, 0x0a, 0x3b, 0x20,
  0x54, 0x4f, 0x44, 0x4f, 0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
  0x20, 0x6e, 0x65, 0x67, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x6e, 0x75,
  0x6d, 0x65, 0x72, 0x73, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20, 0x75, 0x6e,
  0x69, 0x78, 0x20, 0x74, 0x61, 0x69, 0x6c, 0x2f, 0x68, 0x65, 0x61, 0x64,
  0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x20, 0x66, 0x72,
  0x6f, 0x6d, 0x20, 0x65, 0x6e, 0x64, 0x20, 0x62, 0x61, 0x63, 0x6b, 0x77,
  0x61, 0x72, 0x64, 0x73, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x68, 0x65, 0x61,
  0x64, 0x28, 0x6c, 0x73, 0x74, 0x20, 0x6e, 0x29, 0x0a, 0x20, 0x20, 0x22,
  0x46, 0x69, 0x72, 0x73, 0x74, 0x20, 0x6e, 0x20, 0x65, 0x6c, 0x65, 0x6d,
  0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x22, 0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x3c, 0x3d, 0x20, 0x6e, 0x20, 0x30, 0x29, 0x3a, 0x20, 0x4e,
  0x69, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61, 0x74, 0x6f, 0x6d, 0x3f,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x6e, 0x61, 0x6d, 0x65, 0x20,
  0x22, 0x69, 0x6e, 0x64, 0x65, 0x78, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69,
  0x73, 0x74, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x62, 0x6f,
  0x75, 0x6e, 0x64, 0x73, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c,
  0x73, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x61, 0x69,
  0x72, 0x20, 0x66, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x68,
  0x65, 0x61, 0x64, 0x28, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29,
  0x20, 0x28, 0x2d, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x29, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x74, 0x61, 0x69, 0x6c, 0x28, 0x6c, 0x73, 0x74, 0x20,
  0x6e, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4c, 0x61, 0x73, 0x74, 0x20, 0x6e,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f, 0x66,
  0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e,
  0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3c, 0x3d, 0x20, 0x6e, 0x20,
  0x30, 0x29, 0x3a, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x61, 0x74, 0x6f, 0x6d, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x3a, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x72, 0x72, 0x6f, 0x72,
  0x20, 0x27, 0x74, 0x61, 0x69, 0x6c, 0x20, 0x22, 0x69, 0x6e, 0x64, 0x65,
  0x78, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x75,
  0x74, 0x20, 0x6f, 0x66, 0x20, 0x62, 0x6f, 0x75, 0x6e, 0x64, 0x73, 0x22,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20,
  0x74, 0x61, 0x69, 0x6c, 0x28, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74,
  0x29, 0x20, 0x28, 0x2d, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x29, 0x0a, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x28, 0x65, 0x6c,
  0x65, 0x6d, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x3b, 0x20, 0x54, 0x4f,
  0x44, 0x4f, 0x20, 0x73, 0x77, 0x61, 0x70, 0x20, 0x61, 0x72, 0x67, 0x20,
  0x6f, 0x72, 0x64, 0x65, 0x72, 0x3f, 0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e,
  0x64, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x3a,
  0x20, 0x30, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x61, 0x74, 0x6f, 0x6d, 0x3f,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x3a, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72,
  0x28, 0x27, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x20, 0x22, 0x65, 0x78, 0x70,
  0x65, 0x63, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x70, 0x65, 0x72,
  0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x28, 0x3d, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x66, 0x73, 0x74, 0x28,
  0x6c, 0x73, 0x74, 0x29, 0x29, 0x3a, 0x20, 0x28, 0x2b, 0x20, 0x31, 0x20,
  0x63, 0x6f, 0x75, 0x6e, 0x74, 0x28, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x72,
  0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20, 0x63, 0x6f, 0x75, 0x6e,
  0x74, 0x28, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c,
  0x73, 0x74, 0x29, 0x29, 0x0a, 0x3b, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x3b, 0x20, 0x28, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x28, 0x42, 0x69,
  0x6e, 0x61, 0x72, 0x79, 0x28, 0x28, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x3d,
  0x20, 0x66, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x29, 0x20,
  0x2b, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x28, 0x65, 0x6c, 0x65, 0x6d,
  0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x29, 0x29,
  0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x28,
  0x70, 0x72, 0x6f, 0x63, 0x20, 0x61, 0x72, 0x67, 0x73, 0x29, 0x20, 0x3b,
  0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x61,
  0x72, 0x67, 0x73, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x75, 0x6e, 0x20, 0x70,
  0x72, 0x6f, 0x63, 0x65, 0x64, 0x75, 0x72, 0x65, 0x20, 0x77, 0x69, 0x74,
  0x68, 0x20, 0x67, 0x69, 0x76, 0x65, 0x6e, 0x20, 0x61, 0x72, 0x67, 0x75,
  0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a,
  0x20, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x20,
  0x40, 0x28, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x27, 0x71, 0x75, 0x6f, 0x74,
  0x65, 0x20, 0x69, 0x74, 0x29, 0x20, 0x3b, 0x20, 0x70, 0x72, 0x65, 0x76,
  0x65, 0x6e, 0x74, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x61, 0x6e, 0x64,
  0x20, 0x61, 0x72, 0x67, 0x73, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x62,
  0x65, 0x69, 0x6e, 0x67, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74,
  0x65, 0x64, 0x20, 0x74, 0x77, 0x69, 0x63, 0x65, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x50,
  0x61, 0x69, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x61, 0x72, 0x67,
  0x73, 0x29, 0x29, 0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20,
  0x72, 0x65, 0x6e, 0x61, 0x6d, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x66, 0x6f,
  0x72, 0x65, 0x61, 0x63, 0x68, 0x20, 0x28, 0x66, 0x6f, 0x72, 0x29, 0x2c,
  0x20, 0x73, 0x77, 0x61, 0x70, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x61,
  0x6e, 0x64, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44,
  0x4f, 0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x6c, 0x73, 0x74, 0x73, 0x20,
  0x66, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x77, 0x2f, 0x20,
  0x6d, 0x75, 0x6c, 0x74, 0x69, 0x20, 0x61, 0x72, 0x67, 0x75, 0x6d, 0x65,
  0x6e, 0x74, 0x73, 0x2c, 0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x66, 0x6f,
  0x72, 0x20, 0x69, 0x6e, 0x64, 0x65, 0x78, 0x0a, 0x3b, 0x20, 0x5b, 0x6c,
  0x69, 0x6e, 0x65, 0x73, 0x20, 0x30, 0x2e, 0x2e, 0x6c, 0x65, 0x6e, 0x28,
  0x6c, 0x69, 0x6e, 0x65, 0x73, 0x29, 0x5d, 0x20, 0x7c, 0x3e, 0x20, 0x66,
  0x6f, 0x72, 0x65, 0x61, 0x63, 0x68, 0x20, 0x28, 0x6c, 0x69, 0x6e, 0x65,
  0x20, 0x6e, 0x75, 0x6d, 0x29, 0x20, 0x3d, 0x3e, 0x20, 0x70, 0x72, 0x69,
  0x6e, 0x74, 0x6c, 0x6e, 0x28, 0x6e, 0x75, 0x6d, 0x20, 0x22, 0x3a, 0x20,
  0x22, 0x20, 0x6c, 0x69, 0x6e, 0x65, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x6d, 0x61, 0x70, 0x28, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x6c, 0x73, 0x74,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x20,
  0x6e, 0x65, 0x77, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x63, 0x72, 0x65,
  0x61, 0x74, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x61, 0x70, 0x70, 0x6c,
  0x79, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x64, 0x75,
  0x72, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x65,
  0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
  0x65, 0x20, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x22, 0x0a, 0x20, 0x20, 0x3b, 0x20, 0x48, 0x41, 0x43, 0x4b, 0x20, 0x64,
  0x6f, 0x75, 0x62, 0x6c, 0x65, 0x20, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x20,
  0x6d, 0x61, 0x6b, 0x65, 0x73, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x6d,
  0x75, 0x63, 0x68, 0x20, 0x73, 0x6c, 0x6f, 0x77, 0x65, 0x72, 0x20, 0x74,
  0x68, 0x61, 0x6e, 0x20, 0x6e, 0x65, 0x63, 0x65, 0x73, 0x73, 0x61, 0x72,
  0x79, 0x0a, 0x20, 0x20, 0x77, 0x61, 0x6c, 0x6b, 0x28, 0x40, 0x70, 0x72,
  0x6f, 0x63, 0x28, 0x28, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f,
  0x28, 0x69, 0x74, 0x29, 0x20, 0x66, 0x73, 0x74, 0x28, 0x69, 0x74, 0x29,
  0x20, 0x69, 0x74, 0x29, 0x29, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x77, 0x61, 0x6c, 0x6b, 0x28, 0x70, 0x72, 0x6f,
  0x63, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65,
  0x63, 0x75, 0x72, 0x73, 0x69, 0x76, 0x65, 0x6c, 0x79, 0x20, 0x61, 0x70,
  0x70, 0x6c, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x63,
  0x65, 0x64, 0x75, 0x72, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x75, 0x63,
  0x63, 0x65, 0x73, 0x73, 0x69, 0x76, 0x65, 0x20, 0x73, 0x75, 0x62, 0x6c,
  0x69, 0x73, 0x74, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20,
  0x69, 0x6e, 0x70, 0x75, 0x74, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a,
  0x20, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x6e,
  0x69, 0x6c, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x3a, 0x20, 0x4e, 0x69,
  0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28,
  0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50,
  0x61, 0x69, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x28, 0x6c, 0x73, 0x74,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x77, 0x61, 0x6c, 0x6b, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x72,
  0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x28, 0x6c,
  0x73, 0x74, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x63, 0x6f, 0x6e,
  0x76, 0x65, 0x72, 0x74, 0x28, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x6f,
  0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e,
  0x76, 0x65, 0x72, 0x74, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79, 0x20, 0x6d,
  0x65, 0x6d, 0x62, 0x65, 0x72, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x6f,
  0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x69, 0x6e, 0x74, 0x6f, 0x20,
  0x74, 0x6f, 0x22, 0x0a, 0x20, 0x20, 0x6d, 0x61, 0x70, 0x20, 0x40, 0x28,
  0x69, 0x66, 0x20, 0x28, 0x3d, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x69,
  0x74, 0x29, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x74, 0x29, 0x20, 0x6c, 0x73,
  0x74, 0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x61, 0x73,
  0x73, 0x6f, 0x63, 0x20, 0x6d, 0x65, 0x6d, 0x70, 0x20, 0x70, 0x72, 0x6f,
  0x63, 0x65, 0x64, 0x75, 0x72, 0x65, 0x20, 0x65, 0x71, 0x75, 0x69, 0x76,
  0x61, 0x6c, 0x65, 0x6e, 0x74, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x73,
  0x73, 0x6f, 0x63, 0x28, 0x6b, 0x65, 0x79, 0x20, 0x74, 0x61, 0x62, 0x6c,
  0x65, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e,
  0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20,
  0x69, 0x6e, 0x20, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x77, 0x68, 0x65,
  0x72, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6d, 0x61, 0x74,
  0x63, 0x68, 0x65, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6b, 0x65, 0x79,
  0x0a, 0x20, 0x20, 0x49, 0x66, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x66, 0x6f,
  0x75, 0x6e, 0x64, 0x2c, 0x20, 0x72, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x20,
  0x6e, 0x69, 0x6c, 0x22, 0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x74, 0x61, 0x62,
  0x6c, 0x65, 0x29, 0x3a, 0x20, 0x4e, 0x69, 0x6c, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x28, 0x3d, 0x20, 0x6b, 0x65, 0x79, 0x20, 0x66, 0x66, 0x73, 0x74,
  0x28, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x29, 0x29, 0x3a, 0x20, 0x66, 0x73,
  0x74, 0x28, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20, 0x61, 0x73, 0x73, 0x6f, 0x63,
  0x28, 0x6b, 0x65, 0x79, 0x20, 0x72, 0x73, 0x74, 0x28, 0x74, 0x61, 0x62,
  0x6c, 0x65, 0x29, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x66, 0x69,
  0x6c, 0x74, 0x65, 0x72, 0x28, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x6c, 0x73,
  0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4f, 0x6e, 0x6c, 0x79, 0x20, 0x6b,
  0x65, 0x65, 0x70, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73,
  0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x77, 0x68, 0x65,
  0x72, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x69, 0x6e, 0x67, 0x20,
  0x70, 0x72, 0x6f, 0x63, 0x20, 0x72, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x73,
  0x20, 0x74, 0x72, 0x75, 0x65, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f,
  0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x6b, 0x65, 0x65, 0x70, 0x2c, 0x20,
  0x72, 0x65, 0x6d, 0x6f, 0x76, 0x65, 0x2c, 0x20, 0x6d, 0x65, 0x6d, 0x62,
  0x65, 0x72, 0x2c, 0x20, 0x6d, 0x65, 0x6d, 0x70, 0x22, 0x0a, 0x20, 0x20,
  0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6e, 0x6f,
  0x74, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29,
  0x29, 0x3a, 0x20, 0x4e, 0x69, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x70,
  0x72, 0x6f, 0x63, 0x28, 0x66, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29,
  0x29, 0x3a, 0x20, 0x28, 0x50, 0x61, 0x69, 0x72, 0x20, 0x66, 0x73, 0x74,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x20, 0x66, 0x69, 0x6c, 0x74, 0x65, 0x72,
  0x28, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73,
  0x74, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c, 0x73,
  0x65, 0x3a, 0x20, 0x66, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x28, 0x70, 0x72,
  0x6f, 0x63, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x29,
  0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x6b, 0x65, 0x65,
  0x70, 0x2a, 0x20, 0x72, 0x65, 0x6d, 0x6f, 0x76, 0x65, 0x2a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x6b, 0x65, 0x65, 0x70, 0x28, 0x65, 0x6c, 0x65, 0x6d,
  0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x74,
  0x75, 0x72, 0x6e, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x77, 0x69, 0x74,
  0x68, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65,
  0x6e, 0x74, 0x73, 0x20, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x69, 0x6e, 0x67,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f,
  0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x66, 0x69, 0x6c, 0x74, 0x65, 0x72,
  0x2c, 0x20, 0x72, 0x65, 0x6d, 0x6f, 0x76, 0x65, 0x22, 0x0a, 0x20, 0x20,
  0x66, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x20, 0x40, 0x28, 0x3d, 0x20, 0x65,
  0x6c, 0x65, 0x6d, 0x20, 0x69, 0x74, 0x29, 0x20, 0x6c, 0x73, 0x74, 0x0a,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x65, 0x6d, 0x6f, 0x76, 0x65, 0x28,
  0x65, 0x6c, 0x65, 0x6d, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20,
  0x22, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x20, 0x6c, 0x69, 0x73, 0x74,
  0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x65, 0x6c, 0x65,
  0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6d, 0x61, 0x74, 0x63, 0x68, 0x69,
  0x6e, 0x67, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x0a, 0x20, 0x20, 0x41, 0x6c,
  0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x66, 0x69, 0x6c, 0x74,
  0x65, 0x72, 0x2c, 0x20, 0x6b, 0x65, 0x65, 0x70, 0x22, 0x0a, 0x20, 0x20,
  0x66, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x20, 0x40, 0x28, 0x2f, 0x3d, 0x20,
  0x65, 0x6c, 0x65, 0x6d, 0x20, 0x69, 0x74, 0x29, 0x20, 0x6c, 0x73, 0x74,
  0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6d, 0x65, 0x6d, 0x70, 0x28, 0x70,
  0x72, 0x6f, 0x63, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22,
  0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20,
  0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20,
  0x61, 0x66, 0x74, 0x65, 0x72, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20,
  0x74, 0x69, 0x6d, 0x65, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x64, 0x75,
  0x72, 0x65, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x65, 0x64, 0x20, 0x74,
  0x6f, 0x20, 0x65, 0x61, 0x63, 0x68, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
  0x74, 0x20, 0x6e, 0x69, 0x6c, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f,
  0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x6d, 0x65, 0x6d, 0x62, 0x65, 0x72,
  0x2c, 0x20, 0x66, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x22, 0x0a, 0x20, 0x20,
  0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x6e, 0x69, 0x6c,
  0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x3a, 0x20, 0x4e, 0x69, 0x6c, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x28, 0x66, 0x73, 0x74,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x3a, 0x20, 0x6c, 0x73, 0x74, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x65, 0x6c, 0x73, 0x65, 0x3a, 0x20, 0x6d, 0x65,
  0x6d, 0x70, 0x28, 0x70, 0x72, 0x6f, 0x63, 0x20, 0x72, 0x73, 0x74, 0x28,
  0x6c, 0x73, 0x74, 0x29, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6d,
  0x65, 0x6d, 0x62, 0x65, 0x72, 0x28, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x6c,
  0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x74, 0x75, 0x72,
  0x6e, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c,
  0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x61, 0x66, 0x74, 0x65, 0x72,
  0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x6d, 0x61, 0x74, 0x63, 0x68,
  0x69, 0x6e, 0x67, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x0a, 0x20, 0x20, 0x41,
  0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x6d, 0x65, 0x6d,
  0x70, 0x2c, 0x20, 0x66, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x22, 0x0a, 0x20,
  0x20, 0x6d, 0x65, 0x6d, 0x70, 0x20, 0x40, 0x28, 0x3d, 0x20, 0x65, 0x6c,
  0x65, 0x6d, 0x20, 0x69, 0x74, 0x29, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79, 0x70, 0x3f, 0x28,
  0x70, 0x72, 0x6f, 0x63, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20,
  0x22, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0x20, 0x62, 0x6f, 0x6f, 0x6c,
  0x65, 0x61, 0x6e, 0x20, 0x69, 0x66, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79,
  0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x20,
  0x6c, 0x69, 0x73, 0x74, 0x20, 0x70, 0x61, 0x73, 0x73, 0x65, 0x73, 0x20,
  0x70, 0x72, 0x6f, 0x63, 0x22, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x70,
  0x61, 0x69, 0x72, 0x3f, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x69, 0x66, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x28, 0x66, 0x73,
  0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79, 0x70, 0x3f, 0x20, 0x70, 0x72,
  0x6f, 0x63, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x46, 0x61, 0x6c, 0x73, 0x65, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x54, 0x72, 0x75, 0x65, 0x0a, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79, 0x3f, 0x28, 0x65, 0x6c, 0x65,
  0x6d, 0x20, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65,
  0x74, 0x75, 0x72, 0x6e, 0x20, 0x62, 0x6f, 0x6f, 0x6c, 0x65, 0x61, 0x6e,
  0x20, 0x69, 0x66, 0x20, 0x65, 0x76, 0x65, 0x72, 0x79, 0x20, 0x65, 0x6c,
  0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x6c, 0x69, 0x73,
  0x74, 0x20, 0x69, 0x73, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20, 0x74,
  0x6f, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x22, 0x0a, 0x20, 0x20, 0x65, 0x76,
  0x65, 0x72, 0x79, 0x70, 0x3f, 0x20, 0x40, 0x28, 0x3d, 0x20, 0x65, 0x6c,
  0x65, 0x6d, 0x20, 0x69, 0x74, 0x29, 0x20, 0x6c, 0x73, 0x74, 0x0a, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x72, 0x65, 0x76, 0x65, 0x72, 0x73, 0x65, 0x28,
  0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x76, 0x65,
  0x72, 0x73, 0x65, 0x20, 0x6f, 0x72, 0x64, 0x65, 0x72, 0x20, 0x6f, 0x66,
  0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a, 0x20, 0x20, 0x72, 0x65, 0x63,
  0x75, 0x72, 0x20, 0x66, 0x20, 0x28, 0x28, 0x69, 0x6e, 0x20, 0x6c, 0x73,
  0x74, 0x29, 0x20, 0x28, 0x6f, 0x75, 0x74, 0x20, 0x4e, 0x69, 0x6c, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69,
  0x72, 0x3f, 0x28, 0x69, 0x6e, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x66, 0x20, 0x72, 0x73, 0x74, 0x28, 0x69, 0x6e, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x20,
  0x66, 0x73, 0x74, 0x28, 0x69, 0x6e, 0x29, 0x20, 0x6f, 0x75, 0x74, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6f, 0x75, 0x74, 0x0a, 0x0a, 0x3b,
  0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x61, 0x63, 0x63, 0x65, 0x70, 0x74,
  0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x20,
  0x74, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x28, 0x78, 0x20, 0x79,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x41, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x20,
  0x6c, 0x69, 0x73, 0x74, 0x20, 0x79, 0x20, 0x74, 0x6f, 0x20, 0x65, 0x6e,
  0x64, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x78, 0x22,
  0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x78, 0x29, 0x3a, 0x20, 0x79, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x78, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x20, 0x66,
  0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64,
  0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x79, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x3b, 0x20, 0x28, 0x65, 0x6c, 0x73, 0x65, 0x20, 0x28,
  0x50, 0x61, 0x69, 0x72, 0x20, 0x78, 0x20, 0x79, 0x29, 0x29, 0x20, 0x3b,
  0x20, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x20, 0x61, 0x70, 0x70,
  0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x6f, 0x6e, 0x20, 0x6c,
  0x69, 0x73, 0x74, 0x73, 0x20, 0x28, 0x66, 0x6f, 0x72, 0x20, 0x66, 0x6c,
  0x61, 0x74, 0x6d, 0x61, 0x70, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65,
  0x6c, 0x73, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x72,
  0x72, 0x6f, 0x72, 0x20, 0x27, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x64, 0x20,
  0x22, 0x65, 0x78, 0x70, 0x65, 0x63, 0x74, 0x65, 0x64, 0x20, 0x70, 0x72,
  0x6f, 0x70, 0x65, 0x72, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a, 0x0a,
  0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x7a, 0x69, 0x70, 0x20, 0x74,
  0x6f, 0x20, 0x70, 0x72, 0x6f, 0x70, 0x65, 0x72, 0x20, 0x70, 0x61, 0x69,
  0x72, 0x73, 0x20, 0x28, 0x64, 0x65, 0x66, 0x20, 0x7a, 0x69, 0x70, 0x27,
  0x20, 0x28, 0x7a, 0x69, 0x70, 0x20, 0x61, 0x72, 0x67, 0x73, 0x20, 0x28,
  0x6e, 0x69, 0x6c, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x29, 0x29, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x7a, 0x69, 0x70, 0x28, 0x78, 0x20, 0x79, 0x29,
  0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x6d, 0x61, 0x6e, 0x79,
  0x20, 0x61, 0x72, 0x67, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x72, 0x65,
  0x61, 0x74, 0x65, 0x20, 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x20, 0x70,
  0x61, 0x69, 0x72, 0x73, 0x0a, 0x20, 0x20, 0x22, 0x43, 0x72, 0x65, 0x61,
  0x74, 0x65, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x70,
  0x61, 0x69, 0x72, 0x73, 0x20, 0x6d, 0x61, 0x64, 0x65, 0x20, 0x75, 0x70,
  0x20, 0x6f, 0x66, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73,
  0x20, 0x6f, 0x66, 0x20, 0x62, 0x6f, 0x74, 0x68, 0x20, 0x6c, 0x69, 0x73,
  0x74, 0x73, 0x22, 0x0a, 0x20, 0x20, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x61, 0x6e, 0x64, 0x20, 0x6e, 0x69, 0x6c, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x79, 0x29, 0x29,
  0x3a, 0x20, 0x4e, 0x69, 0x6c, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f,
  0x72, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x6e, 0x69,
  0x6c, 0x3f, 0x28, 0x79, 0x29, 0x29, 0x3a, 0x20, 0x28, 0x65, 0x72, 0x72,
  0x6f, 0x72, 0x20, 0x27, 0x7a, 0x69, 0x70, 0x20, 0x22, 0x67, 0x69, 0x76,
  0x65, 0x6e, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73, 0x20, 0x6f, 0x66, 0x20,
  0x75, 0x6e, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20, 0x6c, 0x65, 0x6e, 0x67,
  0x74, 0x68, 0x22, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x28, 0x61, 0x6e,
  0x64, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x70,
  0x61, 0x69, 0x72, 0x3f, 0x28, 0x79, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x20, 0x66, 0x73, 0x74,
  0x28, 0x78, 0x29, 0x20, 0x66, 0x73, 0x74, 0x28, 0x79, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x7a, 0x69, 0x70, 0x20, 0x72,
  0x73, 0x74, 0x28, 0x78, 0x29, 0x20, 0x72, 0x73, 0x74, 0x28, 0x79, 0x29,
  0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x53, 0x74, 0x61, 0x63, 0x6b, 0x73,
  0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x75, 0x73, 0x68, 0x28, 0x73,
  0x74, 0x61, 0x63, 0x6b, 0x20, 0x76, 0x61, 0x6c, 0x29, 0x0a, 0x20, 0x20,
  0x22, 0x41, 0x64, 0x64, 0x20, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x20, 0x74,
  0x6f, 0x20, 0x66, 0x72, 0x6f, 0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73,
  0x74, 0x61, 0x63, 0x6b, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f, 0x20,
  0x73, 0x65, 0x65, 0x3a, 0x20, 0x70, 0x6f, 0x70, 0x2c, 0x20, 0x70, 0x65,
  0x65, 0x6b, 0x22, 0x0a, 0x20, 0x20, 0x50, 0x61, 0x69, 0x72, 0x20, 0x76,
  0x61, 0x6c, 0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x70, 0x6f, 0x70, 0x28, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x29,
  0x0a, 0x20, 0x20, 0x22, 0x47, 0x65, 0x74, 0x20, 0x73, 0x74, 0x61, 0x63,
  0x6b, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x66, 0x69,
  0x72, 0x73, 0x74, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x0a,
  0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20,
  0x70, 0x75, 0x73, 0x68, 0x2c, 0x20, 0x70, 0x65, 0x65, 0x6b, 0x22, 0x0a,
  0x20, 0x20, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x73,
  0x74, 0x61, 0x63, 0x6b, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x72, 0x73,
  0x74, 0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x70, 0x6f, 0x70, 0x20, 0x22,
  0x49, 0x6d, 0x70, 0x72, 0x6f, 0x70, 0x65, 0x72, 0x20, 0x73, 0x74, 0x61,
  0x63, 0x6b, 0x3a, 0x20, 0x22, 0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x65, 0x65, 0x6b, 0x28, 0x73, 0x74,
  0x61, 0x63, 0x6b, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x47, 0x65, 0x74, 0x20,
  0x76, 0x61, 0x6c, 0x75, 0x65, 0x20, 0x61, 0x74, 0x20, 0x66, 0x72, 0x6f,
  0x6e, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a,
  0x20, 0x20, 0x41, 0x6c, 0x73, 0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20,
  0x70, 0x75, 0x73, 0x68, 0x2c, 0x20, 0x70, 0x6f, 0x70, 0x22, 0x0a, 0x20,
  0x20, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f, 0x28, 0x73, 0x74,
  0x61, 0x63, 0x6b, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x66, 0x73, 0x74,
  0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65,
  0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x70, 0x65, 0x65, 0x6b, 0x20, 0x22,
  0x49, 0x6d, 0x70, 0x72, 0x6f, 0x70, 0x65, 0x72, 0x20, 0x73, 0x74, 0x61,
  0x63, 0x6b, 0x3a, 0x20, 0x22, 0x20, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x0a,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x73, 0x77, 0x61, 0x70, 0x28, 0x73, 0x74,
  0x61, 0x63, 0x6b, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x47, 0x65, 0x74, 0x20,
  0x73, 0x74, 0x61, 0x63, 0x6b, 0x20, 0x77, 0x68, 0x65, 0x72, 0x65, 0x20,
  0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x32, 0x20,
  0x76, 0x61, 0x6c, 0x75, 0x65, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x73,
  0x77, 0x61, 0x70, 0x70, 0x65, 0x64, 0x0a, 0x20, 0x20, 0x41, 0x6c, 0x73,
  0x6f, 0x20, 0x73, 0x65, 0x65, 0x3a, 0x20, 0x70, 0x75, 0x73, 0x68, 0x2c,
  0x20, 0x70, 0x6f, 0x70, 0x2c, 0x20, 0x70, 0x65, 0x65, 0x6b, 0x22, 0x0a,
  0x20, 0x20, 0x6c, 0x65, 0x74, 0x20, 0x28, 0x78, 0x3a, 0x20, 0x70, 0x65,
  0x65, 0x6b, 0x28, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x79, 0x3a, 0x20, 0x70, 0x65, 0x65, 0x6b,
  0x28, 0x70, 0x6f, 0x70, 0x28, 0x73, 0x74, 0x61, 0x63, 0x6b, 0x29, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x70, 0x75, 0x73, 0x68, 0x20, 0x70,
  0x75, 0x73, 0x68, 0x28, 0x70, 0x6f, 0x70, 0x28, 0x70, 0x6f, 0x70, 0x28,
  0x73, 0x74, 0x61, 0x63, 0x6b, 0x29, 0x29, 0x20, 0x78, 0x29, 0x20, 0x79,
  0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x74, 0x61, 0x6e,
  0x74, 0x73, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x69, 0x20, 0x28,
  0x2a, 0x20, 0x34, 0x20, 0x61, 0x72, 0x63, 0x74, 0x61, 0x6e, 0x28, 0x31,
  0x2e, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x74, 0x61, 0x75, 0x20,
  0x28, 0x2a, 0x20, 0x32, 0x20, 0x70, 0x69, 0x29, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x65, 0x20, 0x65, 0x78, 0x70, 0x28, 0x31, 0x2e, 0x29, 0x0a, 0x0a,
  0x3b, 0x3b, 0x3b, 0x20, 0x54, 0x65, 0x73, 0x74, 0x73, 0x0a, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x76, 0x65, 0x3f,
  0x28, 0x78, 0x29, 0x20, 0x28, 0x3e, 0x20, 0x78, 0x20, 0x30, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x6e, 0x65, 0x67, 0x61, 0x74, 0x69, 0x76, 0x65,
  0x3f, 0x28, 0x78, 0x29, 0x20, 0x28, 0x3c, 0x20, 0x78, 0x20, 0x30, 0x29,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x7a, 0x65, 0x72, 0x6f, 0x3f, 0x28, 0x78,
  0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x78, 0x20, 0x30,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x3f, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x6d, 0x6f,
  0x64, 0x28, 0x78, 0x20, 0x32, 0x29, 0x20, 0x30, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x6f, 0x64, 0x64, 0x3f, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x3d, 0x20, 0x6d, 0x6f, 0x64, 0x28, 0x78, 0x20,
  0x32, 0x29, 0x20, 0x31, 0x29, 0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x46,
  0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x0a, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x28, 0x2f, 0x3d, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x78, 0x29,
  0x20, 0x28, 0x6e, 0x6f, 0x74, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x28,
  0x3d, 0x20, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x28, 0x2a,
  0x5e, 0x20, 0x78, 0x20, 0x70, 0x29, 0x20, 0x28, 0x2a, 0x20, 0x78, 0x20,
  0x28, 0x5e, 0x20, 0x31, 0x30, 0x20, 0x70, 0x29, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x69, 0x6e, 0x63, 0x28, 0x78, 0x29, 0x20, 0x28, 0x2b, 0x20,
  0x78, 0x20, 0x31, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x64, 0x65, 0x63,
  0x28, 0x78, 0x29, 0x20, 0x28, 0x2d, 0x20, 0x78, 0x20, 0x31, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x74, 0x72, 0x75, 0x6e, 0x63, 0x61, 0x74, 0x65,
  0x28, 0x78, 0x29, 0x20, 0x28, 0x2a, 0x20, 0x66, 0x6c, 0x6f, 0x6f, 0x72,
  0x28, 0x61, 0x62, 0x73, 0x28, 0x78, 0x29, 0x29, 0x20, 0x73, 0x67, 0x6e,
  0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x73, 0x71, 0x72,
  0x28, 0x78, 0x29, 0x20, 0x28, 0x2a, 0x20, 0x78, 0x20, 0x78, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x63, 0x75, 0x62, 0x65, 0x28, 0x78, 0x29, 0x20,
  0x28, 0x2a, 0x20, 0x78, 0x20, 0x28, 0x2a, 0x20, 0x78, 0x20, 0x78, 0x29,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x28, 0x62,
  0x20, 0x70, 0x29, 0x20, 0x28, 0x5e, 0x20, 0x62, 0x20, 0x28, 0x2f, 0x20,
  0x70, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x73, 0x71, 0x72, 0x74,
  0x28, 0x78, 0x29, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x28, 0x78, 0x20, 0x32,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x63, 0x62, 0x72, 0x74, 0x28, 0x78,
  0x29, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x28, 0x78, 0x20, 0x33, 0x29, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x6c, 0x6f, 0x67, 0x62, 0x28, 0x62, 0x20, 0x78,
  0x29, 0x20, 0x28, 0x2f, 0x20, 0x6c, 0x6f, 0x67, 0x28, 0x78, 0x29, 0x20,
  0x6c, 0x6f, 0x67, 0x28, 0x62, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x6c, 0x6f, 0x67, 0x31, 0x30, 0x28, 0x78, 0x29, 0x20, 0x6c, 0x6f, 0x67,
  0x62, 0x28, 0x31, 0x30, 0x2e, 0x20, 0x78, 0x29, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x68, 0x61, 0x6c, 0x66, 0x28, 0x78, 0x29, 0x20, 0x28, 0x2f, 0x20,
  0x78, 0x20, 0x32, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x64, 0x6f, 0x75,
  0x62, 0x6c, 0x65, 0x28, 0x78, 0x29, 0x20, 0x28, 0x2a, 0x20, 0x78, 0x20,
  0x32, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x66, 0x61, 0x63, 0x74,
  0x6f, 0x72, 0x69, 0x61, 0x6c, 0x28, 0x6e, 0x29, 0x0a, 0x20, 0x20, 0x22,
  0x50, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x61,
  0x6c, 0x6c, 0x20, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x76, 0x65, 0x20,
  0x69, 0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x73, 0x20, 0x6c, 0x65, 0x73,
  0x73, 0x20, 0x74, 0x68, 0x61, 0x6e, 0x20, 0x6f, 0x72, 0x20, 0x65, 0x71,
  0x75, 0x61, 0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6e, 0x22, 0x0a, 0x20, 0x20,
  0x72, 0x65, 0x63, 0x75, 0x72, 0x20, 0x6c, 0x6f, 0x6f, 0x70, 0x20, 0x28,
  0x6e, 0x3a, 0x20, 0x6e, 0x20, 0x20, 0x72, 0x65, 0x73, 0x3a, 0x20, 0x31,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x69, 0x66, 0x20, 0x28, 0x3c, 0x3d,
  0x20, 0x6e, 0x20, 0x31, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x72, 0x65, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6c, 0x6f,
  0x6f, 0x70, 0x20, 0x28, 0x2d, 0x20, 0x6e, 0x20, 0x31, 0x29, 0x20, 0x28,
  0x2a, 0x20, 0x6e, 0x20, 0x72, 0x65, 0x73, 0x29, 0x0a, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x67, 0x63, 0x64, 0x28, 0x78, 0x20, 0x79, 0x29, 0x0a, 0x20,
  0x20, 0x22, 0x4c, 0x61, 0x72, 0x67, 0x65, 0x73, 0x74, 0x20, 0x70, 0x6f,
  0x73, 0x69, 0x74, 0x69, 0x76, 0x65, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x67,
  0x65, 0x72, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x64, 0x69, 0x76, 0x69,
  0x64, 0x65, 0x73, 0x20, 0x62, 0x6f, 0x74, 0x68, 0x20, 0x78, 0x20, 0x61,
  0x6e, 0x64, 0x20, 0x79, 0x22, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x28,
  0x3d, 0x20, 0x79, 0x20, 0x30, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x78,
  0x0a, 0x20, 0x20, 0x20, 0x20, 0x67, 0x63, 0x64, 0x28, 0x79, 0x20, 0x6d,
  0x6f, 0x64, 0x28, 0x78, 0x20, 0x79, 0x29, 0x29, 0x0a, 0x0a, 0x3b, 0x20,
  0x54, 0x4f, 0x44, 0x4f, 0x20, 0x6d, 0x76, 0x20, 0x74, 0x6f, 0x20, 0x75,
  0x6e, 0x69, 0x74, 0x73, 0x20, 0x74, 0x69, 0x62, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x64, 0x65, 0x67, 0x28, 0x78, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x43,
  0x6f, 0x6e, 0x76, 0x65, 0x72, 0x74, 0x20, 0x78, 0x20, 0x72, 0x61, 0x64,
  0x69, 0x61, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x64, 0x65, 0x67, 0x72,
  0x65, 0x65, 0x73, 0x22, 0x0a, 0x20, 0x20, 0x2a, 0x20, 0x78, 0x20, 0x28,
  0x2f, 0x20, 0x31, 0x38, 0x30, 0x2e, 0x20, 0x70, 0x69, 0x29, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x72, 0x61, 0x64, 0x28, 0x78, 0x29, 0x0a, 0x20, 0x20,
  0x22, 0x43, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x74, 0x20, 0x78, 0x20, 0x64,
  0x65, 0x67, 0x72, 0x65, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x72, 0x61,
  0x64, 0x69, 0x61, 0x6e, 0x73, 0x22, 0x0a, 0x20, 0x20, 0x2a, 0x20, 0x78,
  0x20, 0x28, 0x2f, 0x20, 0x70, 0x69, 0x20, 0x31, 0x38, 0x30, 0x2e, 0x29,
  0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x54, 0x72, 0x69, 0x67, 0x0a, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x63, 0x73, 0x63, 0x28, 0x78, 0x29, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x28, 0x2f, 0x20, 0x73, 0x69, 0x6e, 0x28, 0x78, 0x29,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x72, 0x63, 0x63, 0x73, 0x63,
  0x28, 0x78, 0x29, 0x20, 0x20, 0x28, 0x2f, 0x20, 0x61, 0x72, 0x63, 0x73,
  0x69, 0x6e, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x63,
  0x73, 0x63, 0x68, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x28, 0x2f,
  0x20, 0x73, 0x69, 0x6e, 0x68, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x61, 0x72, 0x63, 0x63, 0x73, 0x63, 0x68, 0x28, 0x78, 0x29,
  0x20, 0x28, 0x2f, 0x20, 0x61, 0x72, 0x63, 0x73, 0x69, 0x6e, 0x68, 0x28,
  0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x73, 0x65, 0x63, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x2f, 0x20, 0x63, 0x6f,
  0x73, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x72,
  0x63, 0x73, 0x65, 0x63, 0x28, 0x78, 0x29, 0x20, 0x20, 0x28, 0x2f, 0x20,
  0x61, 0x72, 0x63, 0x63, 0x6f, 0x73, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64,
  0x65, 0x66, 0x20, 0x73, 0x65, 0x63, 0x68, 0x28, 0x78, 0x29, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x2f, 0x20, 0x63, 0x6f, 0x73, 0x68, 0x28, 0x78, 0x29,
  0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x72, 0x63, 0x73, 0x65, 0x63,
  0x68, 0x28, 0x78, 0x29, 0x20, 0x28, 0x2f, 0x20, 0x61, 0x72, 0x63, 0x63,
  0x6f, 0x73, 0x68, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x63, 0x6f, 0x74, 0x28, 0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x2f, 0x20, 0x74, 0x61, 0x6e, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x61, 0x72, 0x63, 0x63, 0x6f, 0x74, 0x28, 0x78, 0x29, 0x20,
  0x20, 0x28, 0x2f, 0x20, 0x61, 0x72, 0x63, 0x74, 0x61, 0x6e, 0x28, 0x78,
  0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x63, 0x6f, 0x74, 0x68, 0x28,
  0x78, 0x29, 0x20, 0x20, 0x20, 0x20, 0x28, 0x2f, 0x20, 0x74, 0x61, 0x6e,
  0x68, 0x28, 0x78, 0x29, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x72,
  0x63, 0x63, 0x6f, 0x74, 0x68, 0x28, 0x78, 0x29, 0x20, 0x28, 0x2f, 0x20,
  0x61, 0x72, 0x63, 0x74, 0x61, 0x6e, 0x68, 0x28, 0x78, 0x29, 0x29, 0x0a,
  0x0a, 0x64, 0x65, 0x66, 0x20, 0x61, 0x62, 0x73, 0x28, 0x78, 0x29, 0x0a,
  0x20, 0x20, 0x22, 0x41, 0x62, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x65, 0x20,
  0x76, 0x61, 0x6c, 0x75, 0x65, 0x3a, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x6e,
  0x65, 0x67, 0x61, 0x74, 0x69, 0x76, 0x65, 0x20, 0x76, 0x61, 0x6c, 0x75,
  0x65, 0x20, 0x6f, 0x66, 0x20, 0x78, 0x22, 0x0a, 0x20, 0x20, 0x69, 0x66,
  0x20, 0x28, 0x3e, 0x3d, 0x20, 0x78, 0x20, 0x30, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x78, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x2d, 0x20, 0x78, 0x0a,
  0x64, 0x65, 0x66, 0x20, 0x73, 0x67, 0x6e, 0x28, 0x78, 0x29, 0x0a, 0x20,
  0x20, 0x22, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x78, 0x3a,
  0x20, 0x65, 0x69, 0x74, 0x68, 0x65, 0x72, 0x20, 0x2b, 0x31, 0x2c, 0x20,
  0x2d, 0x31, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x30, 0x22, 0x0a, 0x20, 0x20,
  0x69, 0x66, 0x20, 0x28, 0x3d, 0x20, 0x78, 0x20, 0x30, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x78, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x2f, 0x20, 0x61,
  0x62, 0x73, 0x28, 0x78, 0x29, 0x20, 0x78, 0x0a, 0x64, 0x65, 0x66, 0x20,
  0x6d, 0x61, 0x78, 0x28, 0x61, 0x20, 0x62, 0x29, 0x20, 0x3b, 0x20, 0x54,
  0x4f, 0x44, 0x4f, 0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x61, 0x72, 0x67,
  0x73, 0x0a, 0x20, 0x20, 0x22, 0x4d, 0x61, 0x78, 0x69, 0x6d, 0x75, 0x6d,
  0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x62, 0x5c,
  0x6e, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x6d, 0x61, 0x78, 0x20, 0x31, 0x20,
  0x35, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3b, 0x20, 0x35, 0x0a, 0x20,
  0x20, 0x3e, 0x20, 0x6d, 0x61, 0x78, 0x20, 0x34, 0x2f, 0x35, 0x20, 0x2d,
  0x33, 0x20, 0x20, 0x20, 0x3b, 0x20, 0x34, 0x2f, 0x35, 0x22, 0x0a, 0x20,
  0x20, 0x69, 0x66, 0x20, 0x28, 0x3e, 0x20, 0x61, 0x20, 0x62, 0x29, 0x20,
  0x61, 0x20, 0x62, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x28, 0x6d, 0x69, 0x6e,
  0x20, 0x61, 0x20, 0x62, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4d, 0x69, 0x6e,
  0x69, 0x6d, 0x75, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x61, 0x6e,
  0x64, 0x20, 0x62, 0x5c, 0x6e, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x6d, 0x69,
  0x6e, 0x20, 0x34, 0x32, 0x20, 0x35, 0x34, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x3b, 0x20, 0x34, 0x32, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x6d, 0x69, 0x6e,
  0x20, 0x2e, 0x34, 0x20, 0x31, 0x36, 0x2f, 0x33, 0x20, 0x20, 0x20, 0x3b,
  0x20, 0x2e, 0x34, 0x22, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x28, 0x3c,
  0x20, 0x61, 0x20, 0x62, 0x29, 0x20, 0x61, 0x20, 0x62, 0x0a, 0x64, 0x65,
  0x66, 0x20, 0x63, 0x6c, 0x61, 0x6d, 0x70, 0x28, 0x6e, 0x20, 0x6c, 0x6f,
  0x77, 0x65, 0x72, 0x20, 0x75, 0x70, 0x70, 0x65, 0x72, 0x29, 0x0a, 0x20,
  0x20, 0x22, 0x4c, 0x69, 0x6d, 0x69, 0x74, 0x20, 0x6e, 0x20, 0x62, 0x65,
  0x74, 0x77, 0x65, 0x65, 0x6e, 0x20, 0x6c, 0x6f, 0x77, 0x65, 0x72, 0x20,
  0x61, 0x6e, 0x64, 0x20, 0x75, 0x70, 0x70, 0x65, 0x72, 0x20, 0x62, 0x6f,
  0x75, 0x6e, 0x64, 0x5c, 0x6e, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x63, 0x6c,
  0x61, 0x6d, 0x70, 0x20, 0x31, 0x20, 0x30, 0x20, 0x31, 0x30, 0x20, 0x20,
  0x20, 0x3b, 0x20, 0x31, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x63, 0x6c, 0x61,
  0x6d, 0x70, 0x20, 0x2d, 0x33, 0x20, 0x32, 0x20, 0x34, 0x20, 0x20, 0x20,
  0x3b, 0x20, 0x32, 0x22, 0x0a, 0x20, 0x20, 0x6d, 0x69, 0x6e, 0x20, 0x6d,
  0x61, 0x78, 0x28, 0x6e, 0x20, 0x6c, 0x6f, 0x77, 0x65, 0x72, 0x29, 0x20,
  0x75, 0x70, 0x70, 0x65, 0x72, 0x0a, 0x0a, 0x3b, 0x3b, 0x3b, 0x20, 0x4c,
  0x69, 0x73, 0x74, 0x20, 0x46, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e,
  0x73, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x64, 0x6f, 0x74, 0x28, 0x76,
  0x20, 0x77, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x44, 0x6f, 0x74, 0x20, 0x70,
  0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x20, 0x6f, 0x66, 0x20, 0x6c, 0x69,
  0x73, 0x74, 0x73, 0x20, 0x76, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x77, 0x0a,
  0x20, 0x20, 0x20, 0x53, 0x75, 0x6d, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
  0x65, 0x20, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x73, 0x20, 0x6f,
  0x66, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x20, 0x6f,
  0x66, 0x20, 0x62, 0x6f, 0x74, 0x68, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x73,
  0x22, 0x0a, 0x20, 0x20, 0x69, 0x66, 0x20, 0x76, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x2b, 0x20, 0x28, 0x2a, 0x20, 0x66, 0x73, 0x74, 0x28, 0x76, 0x29,
  0x20, 0x66, 0x73, 0x74, 0x28, 0x77, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x64, 0x6f, 0x74, 0x20, 0x72, 0x73, 0x74, 0x28, 0x76,
  0x29, 0x20, 0x72, 0x73, 0x74, 0x28, 0x77, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x30, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6e, 0x6f, 0x72, 0x6d, 0x28,
  0x76, 0x29, 0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x63, 0x6f,
  0x6d, 0x62, 0x69, 0x6e, 0x65, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61,
  0x62, 0x73, 0x0a, 0x20, 0x20, 0x22, 0x4e, 0x6f, 0x72, 0x6d, 0x61, 0x6c,
  0x69, 0x7a, 0x65, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x22, 0x0a, 0x20, 0x20,
  0x73, 0x71, 0x72, 0x74, 0x20, 0x64, 0x6f, 0x74, 0x28, 0x76, 0x20, 0x76,
  0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x28,
  0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x41, 0x72, 0x69, 0x74,
  0x68, 0x6d, 0x65, 0x74, 0x69, 0x63, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20,
  0x6f, 0x66, 0x20, 0x6c, 0x69, 0x73, 0x74, 0x5c, 0x6e, 0x0a, 0x20, 0x20,
  0x3e, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x5b, 0x31, 0x20, 0x32, 0x20,
  0x33, 0x20, 0x34, 0x20, 0x35, 0x5d, 0x20, 0x20, 0x20, 0x3b, 0x20, 0x33,
  0x0a, 0x20, 0x20, 0x3e, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x20, 0x5b, 0x35,
  0x20, 0x36, 0x20, 0x37, 0x20, 0x38, 0x5d, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x3b, 0x20, 0x31, 0x33, 0x2f, 0x32, 0x22, 0x0a, 0x20, 0x20, 0x72, 0x65,
  0x63, 0x75, 0x72, 0x20, 0x6c, 0x6f, 0x6f, 0x70, 0x20, 0x28, 0x6c, 0x73,
  0x74, 0x3a, 0x20, 0x6c, 0x73, 0x74, 0x20, 0x20, 0x6c, 0x65, 0x6e, 0x3a,
  0x20, 0x30, 0x20, 0x20, 0x61, 0x76, 0x67, 0x3a, 0x20, 0x30, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f,
  0x28, 0x6c, 0x73, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x6c, 0x6f, 0x6f, 0x70, 0x20, 0x72, 0x73, 0x74, 0x28, 0x6c, 0x73, 0x74,
  0x29, 0x20, 0x69, 0x6e, 0x63, 0x28, 0x6c, 0x65, 0x6e, 0x29, 0x20, 0x28,
  0x2b, 0x20, 0x61, 0x76, 0x67, 0x20, 0x66, 0x73, 0x74, 0x28, 0x6c, 0x73,
  0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x2f, 0x20,
  0x61, 0x76, 0x67, 0x20, 0x6c, 0x65, 0x6e, 0x0a, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x72, 0x61, 0x6e, 0x67, 0x65, 0x28, 0x73, 0x74, 0x61, 0x72, 0x74,
  0x20, 0x65, 0x6e, 0x64, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x4c, 0x69, 0x73,
  0x74, 0x20, 0x72, 0x61, 0x6e, 0x67, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x72,
  0x6f, 0x6d, 0x20, 0x73, 0x74, 0x61, 0x72, 0x74, 0x20, 0x74, 0x6f, 0x20,
  0x65, 0x6e, 0x64, 0x2c, 0x20, 0x69, 0x6e, 0x63, 0x6c, 0x75, 0x73, 0x69,
  0x76, 0x65, 0x5c, 0x6e, 0x0a, 0x20, 0x20, 0x3e, 0x20, 0x72, 0x61, 0x6e,
  0x67, 0x65, 0x28, 0x31, 0x20, 0x35, 0x29, 0x20, 0x20, 0x20, 0x3b, 0x20,
  0x5b, 0x31, 0x20, 0x32, 0x20, 0x33, 0x20, 0x34, 0x20, 0x35, 0x5d, 0x0a,
  0x20, 0x20, 0x3e, 0x20, 0x72, 0x61, 0x6e, 0x67, 0x65, 0x20, 0x2d, 0x34,
  0x20, 0x30, 0x20, 0x20, 0x20, 0x3b, 0x20, 0x5b, 0x2d, 0x34, 0x20, 0x2d,
  0x33, 0x20, 0x2d, 0x32, 0x20, 0x2d, 0x31, 0x20, 0x30, 0x5d, 0x22, 0x0a,
  0x20, 0x20, 0x72, 0x65, 0x63, 0x75, 0x72, 0x20, 0x66, 0x20, 0x28, 0x6e,
  0x3a, 0x20, 0x65, 0x6e, 0x64, 0x20, 0x20, 0x72, 0x65, 0x73, 0x3a, 0x20,
  0x4e, 0x69, 0x6c, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x69, 0x66, 0x20,
  0x28, 0x3e, 0x3d, 0x20, 0x6e, 0x20, 0x73, 0x74, 0x61, 0x72, 0x74, 0x29,
  0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x20, 0x64, 0x65, 0x63,
  0x28, 0x6e, 0x29, 0x20, 0x5b, 0x6e, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x72,
  0x65, 0x73, 0x5d, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65,
  0x73, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x6e, 0x65, 0x77, 0x6c, 0x69, 0x6e,
  0x65, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x29, 0x20,
  0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x73, 0x70, 0x61, 0x63, 0x65,
  0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x61, 0x62, 0x20, 0x66, 0x75, 0x6e,
  0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x61, 0x63, 0x63, 0x65,
  0x70, 0x74, 0x20, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x20, 0x6f, 0x66,
  0x20, 0x69, 0x6e, 0x73, 0x74, 0x65, 0x61, 0x64, 0x0a, 0x20, 0x20, 0x69,
  0x66, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x66,
  0x69, 0x6c, 0x65, 0x29, 0x20, 0x6e, 0x69, 0x6c, 0x3f, 0x28, 0x72, 0x73,
  0x74, 0x28, 0x66, 0x69, 0x6c, 0x65, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20,
  0x20, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x20, 0x66, 0x73, 0x74, 0x28,
  0x28, 0x6f, 0x72, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x27, 0x28, 0x73,
  0x74, 0x64, 0x6f, 0x75, 0x74, 0x29, 0x29, 0x29, 0x20, 0x66, 0x69, 0x6c,
  0x65, 0x20, 0x22, 0x5c, 0x6e, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x65,
  0x72, 0x72, 0x6f, 0x72, 0x20, 0x27, 0x6e, 0x65, 0x77, 0x6c, 0x69, 0x6e,
  0x65, 0x20, 0x22, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x7a, 0x65, 0x72, 0x6f,
  0x20, 0x6f, 0x72, 0x20, 0x6f, 0x6e, 0x65, 0x20, 0x66, 0x69, 0x6c, 0x65,
  0x20, 0x63, 0x61, 0x6e, 0x20, 0x62, 0x65, 0x20, 0x67, 0x69, 0x76, 0x65,
  0x6e, 0x22, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x64, 0x69, 0x73, 0x70,
  0x6c, 0x61, 0x79, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x76, 0x61, 0x6c, 0x73,
  0x29, 0x0a, 0x20, 0x20, 0x22, 0x43, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x74,
  0x20, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x73,
  0x74, 0x72, 0x69, 0x6e, 0x67, 0x20, 0x77, 0x68, 0x69, 0x63, 0x68, 0x20,
  0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74, 0x65, 0x73, 0x20, 0x74, 0x6f,
  0x20, 0x69, 0x74, 0x73, 0x65, 0x6c, 0x66, 0x0a, 0x20, 0x20, 0x55, 0x73,
  0x65, 0x64, 0x20, 0x77, 0x68, 0x65, 0x6e, 0x20, 0x64, 0x69, 0x73, 0x70,
  0x6c, 0x61, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x6f, 0x75, 0x74, 0x70, 0x75,
  0x74, 0x20, 0x6f, 0x66, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x22, 0x0a, 0x20,
  0x20, 0x61, 0x70, 0x70, 0x6c, 0x79, 0x20, 0x53, 0x74, 0x72, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x6d, 0x61, 0x70, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x40, 0x28, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67,
  0x3f, 0x28, 0x69, 0x74, 0x29, 0x3a, 0x20, 0x28, 0x53, 0x74, 0x72, 0x20,
  0x22, 0x5c, 0x22, 0x22, 0x20, 0x69, 0x74, 0x20, 0x22, 0x5c, 0x22, 0x22,
  0x29, 0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x69, 0x6e, 0x73,
  0x65, 0x72, 0x74, 0x20, 0x65, 0x73, 0x63, 0x61, 0x70, 0x65, 0x20, 0x63,
  0x68, 0x61, 0x72, 0x61, 0x63, 0x74, 0x65, 0x72, 0x73, 0x20, 0x66, 0x6f,
  0x72, 0x20, 0x6e, 0x6f, 0x6e, 0x2d, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x61,
  0x62, 0x6c, 0x65, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x74, 0x72, 0x75, 0x65, 0x3f, 0x28, 0x69, 0x74, 0x29, 0x3a,
  0x20, 0x20, 0x20, 0x28, 0x53, 0x74, 0x72, 0x20, 0x69, 0x74, 0x29, 0x20,
  0x3b, 0x20, 0x64, 0x6f, 0x6e, 0x27, 0x74, 0x20, 0x71, 0x75, 0x6f, 0x74,
  0x65, 0x20, 0x54, 0x72, 0x75, 0x65, 0x20, 0x73, 0x79, 0x6d, 0x62, 0x6f,
  0x6c, 0x20, 0x73, 0x69, 0x6e, 0x63, 0x65, 0x20, 0x69, 0x74, 0x27, 0x73,
  0x20, 0x73, 0x65, 0x6c, 0x66, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x75, 0x61,
  0x74, 0x69, 0x6e, 0x67, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x73, 0x79, 0x6d, 0x62, 0x6f, 0x6c, 0x3f, 0x28, 0x69,
  0x74, 0x29, 0x3a, 0x20, 0x28, 0x53, 0x74, 0x72, 0x20, 0x22, 0x27, 0x22,
  0x20, 0x69, 0x74, 0x29, 0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20,
  0x69, 0x66, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x73, 0x20,
  0x6e, 0x6f, 0x6e, 0x20, 0x73, 0x79, 0x6d, 0x62, 0x6f, 0x6c, 0x20, 0x63,
  0x68, 0x61, 0x72, 0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x20, 0x61, 0x73,
  0x20, 0x65, 0x78, 0x70, 0x6c, 0x69, 0x63, 0x69, 0x74, 0x20, 0x73, 0x79,
  0x6d, 0x20, 0x73, 0x79, 0x6e, 0x74, 0x61, 0x78, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x61, 0x69, 0x72, 0x3f,
  0x28, 0x69, 0x74, 0x29, 0x3a, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x53, 0x74, 0x72, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x22, 0x5b, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x69, 0x73, 0x70, 0x6c,
  0x61, 0x79, 0x28, 0x66, 0x73, 0x74, 0x28, 0x69, 0x74, 0x29, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x28, 0x6f, 0x72, 0x20, 0x28, 0x77, 0x61, 0x6c, 0x6b, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x46, 0x75, 0x6e, 0x63,
  0x28, 0x28, 0x78, 0x29, 0x20, 0x28, 0x69, 0x66, 0x20, 0x70, 0x61, 0x69,
  0x72, 0x3f, 0x28, 0x78, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x53, 0x74, 0x72, 0x28, 0x22, 0x20, 0x22, 0x20, 0x64, 0x69, 0x73,
  0x70, 0x6c, 0x61, 0x79, 0x28, 0x66, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x74, 0x72,
  0x28, 0x22, 0x20, 0x2e, 0x2e, 0x2e, 0x20, 0x22, 0x20, 0x64, 0x69, 0x73,
  0x70, 0x6c, 0x61, 0x79, 0x28, 0x78, 0x29, 0x29, 0x29, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x73, 0x74, 0x28, 0x69,
  0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x22,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x22, 0x5d, 0x22, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x72,
  0x64, 0x3f, 0x28, 0x69, 0x74, 0x29, 0x3a, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x53, 0x74, 0x72,
  0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x22, 0x7b, 0x20, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x6f,
  0x72, 0x20, 0x28, 0x6d, 0x61, 0x70, 0x20, 0x46, 0x75, 0x6e, 0x63, 0x28,
  0x28, 0x78, 0x29, 0x20, 0x53, 0x74, 0x72, 0x28, 0x66, 0x73, 0x74, 0x28,
  0x78, 0x29, 0x20, 0x22, 0x3a, 0x20, 0x22, 0x20, 0x64, 0x69, 0x73, 0x70,
  0x6c, 0x61, 0x79, 0x28, 0x72, 0x73, 0x74, 0x28, 0x78, 0x29, 0x29, 0x20,
  0x22, 0x20, 0x22, 0x29, 0x29, 0x20, 0x72, 0x65, 0x63, 0x6f, 0x72, 0x64,
  0x73, 0x28, 0x69, 0x74, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x22, 0x22, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x7d, 0x22, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x6c,
  0x73, 0x65, 0x3a, 0x20, 0x28, 0x53, 0x74, 0x72, 0x20, 0x69, 0x74, 0x29,
  0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x76, 0x61, 0x6c, 0x73,
  0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x28,
  0x2e, 0x2e, 0x2e, 0x20, 0x73, 0x74, 0x72, 0x29, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x28, 0x27, 0x73, 0x74, 0x64,
  0x6f, 0x75, 0x74, 0x20, 0x4e, 0x69, 0x6c, 0x20, 0x2e, 0x2e, 0x2e, 0x20,
  0x73, 0x74, 0x72, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x72, 0x69,
  0x6e, 0x74, 0x6c, 0x6e, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x73, 0x74, 0x72,
  0x29, 0x20, 0x20, 0x20, 0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x28, 0x2e,
  0x2e, 0x2e, 0x20, 0x73, 0x74, 0x72, 0x29, 0x20, 0x28, 0x6e, 0x65, 0x77,
  0x6c, 0x69, 0x6e, 0x65, 0x29, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x70, 0x72,
  0x69, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x73,
  0x74, 0x72, 0x29, 0x20, 0x20, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x28,
  0x27, 0x73, 0x74, 0x64, 0x65, 0x72, 0x72, 0x20, 0x4e, 0x69, 0x6c, 0x20,
  0x2e, 0x2e, 0x2e, 0x20, 0x73, 0x74, 0x72, 0x29, 0x0a, 0x64, 0x65, 0x66,
  0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x6c, 0x6e, 0x28,
  0x2e, 0x2e, 0x2e, 0x20, 0x73, 0x74, 0x72, 0x29, 0x20, 0x70, 0x72, 0x69,
  0x6e, 0x74, 0x65, 0x72, 0x72, 0x28, 0x2e, 0x2e, 0x2e, 0x20, 0x73, 0x74,
  0x72, 0x29, 0x20, 0x28, 0x6e, 0x65, 0x77, 0x6c, 0x69, 0x6e, 0x65, 0x29,
  0x0a, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x25, 0x20, 0x73,
  0x74, 0x61, 0x63, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x72, 0x65, 0x76,
  0x20, 0x72, 0x65, 0x73, 0x2c, 0x20, 0x25, 0x25, 0x20, 0x3d, 0x20, 0x66,
  0x73, 0x74, 0x20, 0x25, 0x0a, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20,
  0x61, 0x72, 0x67, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x6f, 0x20,
  0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65,
  0x2c, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x2c, 0x20, 0x64, 0x69, 0x73, 0x70,
  0x6c, 0x61, 0x79, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x28, 0x72, 0x65, 0x70,
  0x6c, 0x29, 0x0a, 0x20, 0x20, 0x22, 0x52, 0x65, 0x61, 0x64, 0x2c, 0x20,
  0x65, 0x76, 0x61, 0x6c, 0x75, 0x61, 0x74, 0x65, 0x2c, 0x20, 0x70, 0x72,
  0x69, 0x6e, 0x74, 0x2c, 0x20, 0x6c, 0x6f, 0x6f, 0x70, 0x0a, 0x20, 0x20,
  0x54, 0x6f, 0x20, 0x65, 0x78, 0x69, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x65,
  0x72, 0x20, 0x71, 0x75, 0x69, 0x74, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72,
  0x65, 0x73, 0x73, 0x20, 0x43, 0x54, 0x52, 0x4c, 0x2d, 0x44, 0x22, 0x0a,
  0x20, 0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x20, 0x22, 0x3e, 0x3e, 0x3e,
  0x20, 0x22, 0x0a, 0x20, 0x20, 0x6c, 0x65, 0x74, 0x20, 0x28, 0x65, 0x78,
  0x70, 0x72, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x73, 0x65, 0x28, 0x72, 0x65,
  0x61, 0x64, 0x28, 0x29, 0x29, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x75,
  0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x28, 0x3d, 0x20, 0x65, 0x78, 0x70,
  0x72, 0x20, 0x27, 0x71, 0x75, 0x69, 0x74, 0x29, 0x0a, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x3b, 0x20, 0x54, 0x4f, 0x44, 0x4f, 0x20, 0x70, 0x75,
  0x73, 0x68, 0x20, 0x61, 0x6e, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x74,
  0x61, 0x63, 0x6b, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x70, 0x75,
  0x74, 0x73, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6c, 0x65, 0x74,
  0x20, 0x28, 0x61, 0x6e, 0x73, 0x3a, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x28,
  0x65, 0x78, 0x70, 0x72, 0x29, 0x29, 0x20, 0x3b, 0x20, 0x65, 0x6e, 0x76,
  0x7b, 0x20, 0x3a, 0x25, 0x20, 0x25, 0x25, 0x3a, 0x20, 0x66, 0x73, 0x74,
  0x28, 0x25, 0x29, 0x20, 0x7d, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x75, 0x6e, 0x6c, 0x65, 0x73, 0x73, 0x20, 0x76, 0x6f, 0x69,
  0x64, 0x3f, 0x28, 0x61, 0x6e, 0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x6c,
  0x6e, 0x20, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x28, 0x61, 0x6e,
  0x73, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28,
  0x72, 0x65, 0x70, 0x6c, 0x29, 0x0a, 0x0a, 0x64, 0x65, 0x66, 0x20, 0x28,
  0x72, 0x65, 0x70, 0x6c, 0x2d, 0x73, 0x69, 0x6d, 0x70, 0x6c, 0x65, 0x29,
  0x0a, 0x20, 0x20, 0x22, 0x53, 0x69, 0x6d, 0x70, 0x6c, 0x65, 0x20, 0x52,
  0x45, 0x50, 0x4c, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x66, 0x61, 0x63,
  0x65, 0x2c, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x64, 0x65, 0x70, 0x65,
  0x6e, 0x64, 0x65, 0x6e, 0x63, 0x69, 0x65, 0x73, 0x20, 0x61, 0x72, 0x65,
  0x20, 0x62, 0x75, 0x69, 0x6c, 0x74, 0x69, 0x6e, 0x20, 0x66, 0x75, 0x6e,
  0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2c, 0x20, 0x6e, 0x6f, 0x20, 0x65,
  0x65, 0x76, 0x6f, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x0a, 0x20, 0x20, 0x54,
  0x6f, 0x20, 0x65, 0x78, 0x69, 0x74, 0x20, 0x65, 0x6e, 0x74, 0x65, 0x72,
  0x20, 0x71, 0x75, 0x69, 0x74, 0x20, 0x6f, 0x72, 0x20, 0x70, 0x72, 0x65,
  0x73, 0x73, 0x20, 0x43, 0x54, 0x52, 0x4c, 0x2d, 0x44, 0x0a, 0x20, 0x20,
  0x53, 0x65, 0x65, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x20, 0x66, 0x6f, 0x72,
  0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x61, 0x64, 0x76, 0x61, 0x6e, 0x63,
  0x65, 0x64, 0x20, 0x52, 0x45, 0x50, 0x4c, 0x20, 0x77, 0x69, 0x74, 0x68,
  0x20, 0x70, 0x72, 0x65, 0x74, 0x74, 0x69, 0x65, 0x72, 0x20, 0x70, 0x72,
  0x69, 0x6e, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x0a, 0x20, 0x20, 0x77, 0x72,
  0x69, 0x74, 0x65, 0x20, 0x27, 0x73, 0x74, 0x64, 0x6f, 0x75, 0x74, 0x20,
  0x4e, 0x69, 0x6c, 0x20, 0x22, 0x3e, 0x20, 0x22, 0x0a, 0x20, 0x20, 0x28,
  0x46, 0x75, 0x6e, 0x63, 0x20, 0x28, 0x65, 0x78, 0x70, 0x72, 0x29, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x63, 0x6f, 0x6e, 0x64, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x28, 0x3d, 0x20, 0x65, 0x78,
  0x70, 0x72, 0x20, 0x27, 0x71, 0x75, 0x69, 0x74, 0x29, 0x29, 0x0a, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x65, 0x6c, 0x73, 0x65, 0x0a,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x77, 0x72, 0x69,
  0x74, 0x65, 0x28, 0x27, 0x73, 0x74, 0x64, 0x6f, 0x75, 0x74, 0x20, 0x4e,
  0x69, 0x6c, 0x20, 0x65, 0x76, 0x61, 0x6c, 0x28, 0x65, 0x78, 0x70, 0x72,
  0x29, 0x20, 0x22, 0x5c, 0x6e, 0x22, 0x29, 0x0a, 0x20, 0x20, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x20, 0x28, 0x72, 0x65, 0x70, 0x6c, 0x2d, 0x73,
  0x69, 0x6d, 0x70, 0x6c, 0x65, 0x29, 0x29, 0x29, 0x29, 0x28, 0x70, 0x61,
  0x72, 0x73, 0x65, 0x28, 0x72, 0x65, 0x61, 0x64, 0x28, 0x29, 0x29, 0x29,
  0x0a
, 0x00};
